package xml

import (
	"bytes"
	coral "coral/assembly"
	"coral/encoding"
	"encoding/xml"
	"io"
	"strings"
)

var Decoder encoding.Decoder = encoding.DecoderFunc(Decode)

// Converts an XML token into an lowercase token (effectively
// making the xml case insensitive) only element names and attribute
// names are changed, values are not touched, e.g.
//   <Integer Target="CamelCase">CamelCase</Integer>
// are changed into:
//   <integer target="CamelCase">CamelCase</integer>
func makeLowerCaseToken(t xml.Token) xml.Token {
	switch v := t.(type) {
	case xml.StartElement:
		copy := v.Copy()
		copy.Name.Local = strings.ToLower(v.Name.Local)
		for index, _ := range copy.Attr {
			copy.Attr[index].Name.Local = strings.ToLower(copy.Attr[index].Name.Local)
		}

		return copy

	case xml.EndElement:
		var copy xml.EndElement
		copy.Name.Space = v.Name.Space
		copy.Name.Local = strings.ToLower(v.Name.Local)

		return copy

	default:
		return t
	}
}

// parses the xml from 'rdr' and converts all the
// tokens into lowercase tokens using makeLowerCaseToken
// and encodes the lower case tokens into a new reader
// which is returned (and can be fed into xml.Decoder again)
func preprocessXml(rdr io.Reader) io.Reader {
	var result bytes.Buffer
	decoder := xml.NewDecoder(rdr)
	encoder := xml.NewEncoder(&result)

	for {
		originalToken, err := decoder.Token()
		if err != nil && err != io.EOF {
			panic(err)
		}

		if originalToken == nil {
			break
		}

		newToken := makeLowerCaseToken(originalToken)

		encoder.EncodeToken(newToken)
	}
	encoder.Flush()

	return bytes.NewReader(result.Bytes())
}

func Decode(rdrs []io.Reader) (asms coral.Assemblies) {
	definitions := make(map[string]*Definition, 0)
	decoders := make([]*xml.Decoder, len(rdrs))
	for i, rdr := range rdrs {
		// The Java Coral generator is case-insensitive so
		// we need to preprocess the XML by converting
		// all XML element names and attribute names into
		// lowercase (but we do not touch values)
		decoders[i] = xml.NewDecoder(preprocessXml(rdr))
	}

	//Step 1: Decode everything
	for _, decoder := range decoders {
		var newDef *Definition = new(Definition)
		err := decoder.Decode(newDef)
		if err != nil && err != io.EOF {
			panic(err)
		}
		if oldDef := definitions[newDef.Assembly]; oldDef != nil {
			err := newDef.Merge(oldDef)
			if nil != err {
				panic(err)
			}
		}
		definitions[newDef.Assembly] = newDef
	}

	//Step 2: The coral xml validator doesn't error on exact duplicates, remove those
	for _, definition := range definitions {
		definition.RemoveExactDuplicates()
	}

	//Step 3: Convert everything into the IR
	asms = make(coral.Assemblies, 0, len(definitions))
	for _, def := range definitions {
		asm := def.ToAssembly()
		asms = append(asms, asm)
	}
	return
}
