package xml

//definition struct exists in xml_structs.go

import (
	coral "coral/assembly"
	"coral/shape"
	"fmt"
	"reflect"
	"strings"
)

// The Coral Validator doesn't error if two xml nodes are exactly the same.
// We need to remove the duplicates so we don't have duplicate code.
func (def *Definition) RemoveExactDuplicates() {
	value := reflect.ValueOf(def).Elem()
	for i := 0; i < value.NumField(); i++ {
		field := value.Field(i)
		tipe := reflect.TypeOf(field.Interface())
		if field.Kind() == reflect.Slice && tipe.Elem().Implements(uniqueType) {
			set := make(map[string]Unique, 0)
			for j := 0; j < field.Len(); j++ {
				unique := field.Index(j).Interface().(Unique)
				old := set[unique.Unique()]
				if old != nil {
					if reflect.DeepEqual(old, unique) {
						fmt.Printf("Found Duplicate Item %s\n", unique.Unique())
					} else {
						panic("Duplicate items found. This should have been caught by the coral validator")
					}
				} else {
					set[unique.Unique()] = unique
				}
			}
			//Erase the old slice
			field.Set(reflect.MakeSlice(tipe, 0, len(set)))
			for _, value := range set {
				field.Set(reflect.Append(field, reflect.ValueOf(value)))
			}
		}
	}
}

func (def *Definition) Merge(other *Definition) error {
	if def.Assembly == "" {
		def.Assembly = other.Assembly
	} else if other.Assembly != def.Assembly {
		return fmt.Errorf("Cannot merge definitions with different assemblies. Mine: %s Theirs %s", def.Assembly, other.Assembly)
	}
	if def.Version == "" {
		def.Version = other.Version
	} else if other.Version != def.Version {
		return fmt.Errorf("Definition versions do not match. Mine %s Theirs %s", def.Version, other.Version)
	}

	defValue := reflect.ValueOf(def).Elem()
	otherValue := reflect.ValueOf(other).Elem()
	for i := 0; i < otherValue.NumField(); i++ {
		myField := defValue.Field(i)
		theirField := otherValue.Field(i)
		if myField.Kind() == reflect.Slice {
			myField.Set(reflect.AppendSlice(myField, theirField))
		}
	}
	return nil
}

func (def *Definition) ToAssembly() (asm coral.Assembly) {
	asm.Namespace = def.Assembly

	asm.Shapes = []shape.Interface{}
	asm.Services = []shape.Service{}
	shapeDocs := map[shape.Reference]string{}
	memberDocs := map[shape.Reference]map[string]string{}
	exceptions := map[shape.Reference]struct{}{}
	restEndpoints := map[shape.Reference][]shape.RestEndpoint{}

	//TODO: Adding a new hash up here for every trait is a bit unwieldy. Refactor the trait handling code
	addShape := func(s shape.Interface) {
		if doc, ok := shapeDocs[s.Name()]; ok {
			s.SetDoc(doc)
			delete(shapeDocs, s.Name())
		}
		if docs, ok := memberDocs[s.Name()]; ok {
			//Have something referencing a member, so it has to be a structure
			if s, ok := s.(*shape.Structure); ok {
				for name, member := range s.Members {
					if doc, ok := docs[name]; ok {
						member.Doc = doc
					}
				}
			} else {
				fmt.Printf("Warning: Had documentation referencing members of a non-structure shape: %v", s.Name())
			}
			delete(memberDocs, s.Name())
		}
		if _, ok := exceptions[s.Name()]; ok {
			if s, ok := s.(*shape.Structure); ok {
				s.IsException = true
			} else {
				fmt.Printf("Warning: Had Exception referencing a non-structure shape: %v", s.Name())
			}
			delete(exceptions, s.Name())
		}
		asm.Shapes = append(asm.Shapes, s)
	}
	addCoralShape := func(s coralShape) {
		addShape(s.toIR(def))
	}
	addOperation := func(o *shape.Operation) {
		addShape(o)
		if endpoints, foundEndpoints := restEndpoints[o.Name()]; foundEndpoints {
			o.Endpoints = endpoints
		}
	}

	for _, doc := range def.Documentation {
		ref := def.parseReference(doc.Target.Target)
		if memberRef, ok := ref.(shape.MemberReference); ok {
			baseRef := shape.NewReference(ref.Assembly(), ref.Assembly(), ref.Name())
			docs := memberDocs[baseRef]
			if docs == nil {
				docs = map[string]string{}
				memberDocs[baseRef] = docs
			}
			docs[memberRef.Member()] = strings.TrimSpace(doc.Text)
		} else {
			shapeDocs[ref] = strings.TrimSpace(doc.Text)
		}
	}

	for _, exception := range def.Exception {
		ref := def.parseReference(exception.Target.Target)
		exceptions[ref] = struct{}{}
	}

	for _, endpoint := range def.Http {
		ref := def.parseReference(endpoint.Target.Target)
		restEndpoint := shape.RestEndpoint{Verb: endpoint.Verb.Value, Uri: endpoint.URI.Value}
		restEndpoints[ref] = append(restEndpoints[ref], restEndpoint)
	}

	for _, blob := range def.Blob {
		addCoralShape(blob)
	}

	for _, boolean := range def.Boolean {
		addCoralShape(boolean)
	}

	for _, b := range def.Byte {
		addCoralShape(b)
	}

	for _, time := range def.Timestamp {
		addCoralShape(time)
	}

	for _, char := range def.Character {
		addCoralShape(char)
	}

	for _, dbl := range def.Double {
		addCoralShape(dbl)
	}

	for _, f := range def.Float {
		addCoralShape(f)
	}

	for _, i := range def.Integer {
		addCoralShape(i)
	}

	for _, l := range def.Long {
		addCoralShape(l)
	}

	for _, s := range def.Short {
		addCoralShape(s)
	}

	for _, s := range def.String {
		addCoralShape(s)
	}

	for _, b := range def.BigInteger {
		addCoralShape(b)
	}

	for _, b := range def.BigDecimal {
		addCoralShape(b)
	}

	for _, b := range def.Envelope {
		addCoralShape(b)
	}

	for _, s := range def.Structure {
		addCoralShape(s)
	}

	for _, m := range def.Map {
		addCoralShape(m)
	}

	for _, l := range def.List {
		addCoralShape(l)
	}

	operations := make(map[shape.Reference]*shape.Operation, 0)
	for _, o := range def.Operation {
		op, _ := o.toIR(def).(*shape.Operation)
		operations[op.Name()] = op
		addOperation(op)
	}

	for _, s := range def.Service {
		svc := s.toIR(def, operations).(*shape.Service)
		asm.Services = append(asm.Services, *svc)
		addShape(svc)
	}
	for ref, members := range memberDocs {
		for member, doc := range members {
			memberRef := shape.NewMemberReference(ref.Assembly(), ref.Assembly(), ref.Name(), member)
			if memberRef == nil {
				panic("Nil!")
			}
			shapeDocs[memberRef] = doc
		}
	}
	for key := range shapeDocs {
		fmt.Printf("Had documentation referencing %v, but no such reference exists\n", key)
	}
	for key := range exceptions {
		fmt.Printf("Had exception referencing %v, but no such reference exists\n", key)
	}
	return
}

type coralShape interface {
	toIR(def *Definition) shape.Interface
}

func (b Blob) toIR(def *Definition) shape.Interface {
	return shape.NewBlob(def.parseReference(b.Name))
}
func (b Boolean) toIR(def *Definition) shape.Interface {
	return shape.NewBoolean(def.parseReference(b.Name))
}
func (b Byte) toIR(def *Definition) shape.Interface {
	return shape.NewByte(def.parseReference(b.Name))
}
func (t Timestamp) toIR(def *Definition) shape.Interface {
	return shape.NewTimestamp(def.parseReference(t.Name))
}
func (c Character) toIR(def *Definition) shape.Interface {
	return shape.NewCharacter(def.parseReference(c.Name))
}
func (d Double) toIR(def *Definition) shape.Interface {
	return shape.NewDouble(def.parseReference(d.Name))
}
func (f Float) toIR(def *Definition) shape.Interface {
	return shape.NewFloat(def.parseReference(f.Name))
}
func (i Integer) toIR(def *Definition) shape.Interface {
	return shape.NewInteger(def.parseReference(i.Name))
}
func (l Long) toIR(def *Definition) shape.Interface {
	return shape.NewLong(def.parseReference(l.Name))
}
func (s Short) toIR(def *Definition) shape.Interface {
	return shape.NewShort(def.parseReference(s.Name))
}
func (s String) toIR(def *Definition) shape.Interface {
	return shape.NewString(def.parseReference(s.Name))
}
func (b BigInteger) toIR(def *Definition) shape.Interface {
	return shape.NewBigInteger(def.parseReference(b.Name))
}
func (b BigDecimal) toIR(def *Definition) shape.Interface {
	return shape.NewBigDecimal(def.parseReference(b.Name))
}
func (e Envelope) toIR(def *Definition) shape.Interface {
	return shape.NewEnvelope(def.parseReference(e.Name))
}

func (s Structure) toIR(def *Definition) shape.Interface {
	iface := shape.NewStructure(def.parseReference(s.Name))
	iface.Abstract = s.Abstract
	if s.IsA != "" {
		ref := def.parseReference(s.IsA)
		iface.IsA = &ref
	}
	for _, member := range s.Member {
		iface.Members[member.Name] = shape.StructureMember{Reference: def.parseReference(member.Target.Target)}
	}
	return iface
}

func (l List) toIR(def *Definition) shape.Interface {
	iface := shape.NewList(def.parseReference(l.Name))
	iface.Member = def.parseReference(l.Member.Target)
	return iface
}

func (m Map) toIR(def *Definition) shape.Interface {
	iface := shape.NewMap(def.parseReference(m.Name))
	iface.Key = def.parseReference(m.Key.Target)
	iface.Value = def.parseReference(m.Value.Target)
	return iface

}

func (s Service) toIR(def *Definition, operations map[shape.Reference]*shape.Operation) shape.Interface {
	serviceRef := def.parseReference(s.Name)
	iface := shape.NewService(serviceRef)
	//Remove duplicate operations by using a map
	serviceOps := map[Target]bool{}
	for _, operation := range s.Operation {
		serviceOps[operation] = true
	}
	for operation := range serviceOps {
		ref := def.parseReference(operation.Target)
		shp, ok := operations[ref]
		if ok {
			iface.Operations = append(iface.Operations, shp)
			shp.Services = append(shp.Services, serviceRef)
		} else if def.Assembly == ref.Assembly() { // missing expected reference in _this_ assembly
			panic(fmt.Errorf("Could not find Operation %v", ref))
		}
	}
	return iface
}

func (o Operation) toIR(def *Definition) shape.Interface {
	iface := shape.NewOperation(def.parseReference(o.Name))
	if o.Input.Target != "" {
		ref := def.parseReference(o.Input.Target)
		iface.Input = ref
	}
	if o.Output.Target != "" {
		ref := def.parseReference(o.Output.Target)
		iface.Output = ref
	}
	if o.Error != nil {
		for _, err := range o.Error {
			iface.Errors = append(iface.Errors, def.parseReference(err.Target))
		}
	}
	return iface
}

// Possible Syntaxes:
// assembly-name#shape-name
// shape-name
// assembly-name#shape-name$member
// shape-name$member
//
// Note: if the reference uses the assembly-name#shape-name syntax scheme, it's possible that the assembly name parsed
// from here does not match the assembly from the Definition. This may be intentional, but may also be due to a typo or
// XML transformation error (Codigo XSD or WSDLs to Coral XML). We'll store both the parsed assembly name and the
// assembly name from the definition, in case `LookupShape` from `coral\assembly\assembly.go` fails to match a shape
// reference's assembly directly to the Assembly's name.
func (def *Definition) parseReference(s string) shape.Reference {
	fileAsm := def.Assembly
	asm := def.Assembly
	name := s
	if split := strings.SplitN(s, "#", 2); len(split) == 2 {
		asm = split[0]
		name = split[1]
	}
	if split := strings.SplitN(s, "$", 2); len(split) == 2 {
		name = split[0]
		member := split[1]
		return shape.NewMemberReference(fileAsm, asm, name, member)
	}
	return shape.NewReference(fileAsm, asm, name)
}
