package xml

type Definition struct {
	Version  string `xml:"version,attr"`
	Assembly string `xml:"assembly,attr"`

	Service   []Service   `xml:"service"`
	Operation []Operation `xml:"operation"`

	Blob       []Blob       `xml:"blob"`
	Boolean    []Boolean    `xml:"boolean"`
	Byte       []Byte       `xml:"byte"`
	Timestamp  []Timestamp  `xml:"timestamp"`
	Character  []Character  `xml:"character"`
	Double     []Double     `xml:"double"`
	Float      []Float      `xml:"float"`
	Integer    []Integer    `xml:"integer"`
	Long       []Long       `xml:"long"`
	Short      []Short      `xml:"short"`
	String     []String     `xml:"string"`
	BigInteger []BigInteger `xml:"biginteger"`
	BigDecimal []BigDecimal `xml:"bigdecimal"`
	Envelope   []Envelope   `xml:"envelope"`

	Structure []Structure `xml:"structure"`
	List      []List      `xml:"list"`
	Map       []Map       `xml:"map"`

	Length   []Length   `xml:"length"`
	Required []Required `xml:"required"`
	Enum     []Enum     `xml:"enum"`
	Pattern  []Pattern  `xml:"pattern"`
	Range    []Range    `xml:"range"`

	Idempotent    []Idempotent    `xml:"idempotent"`
	Exception     []Exception     `xml:"exception"`
	Fault         []Fault         `xml:"fault"`
	Box           []Box           `xml:"box"`
	XmlName       []XmlName       `xml:"xmlname"`
	XmlNamespace  []XmlNamespace  `xml:"xmlnamespace"`
	XmlArtificial []XmlArtificial `xml:"xmlartificial"`
	XmlOrder      []XmlOrder      `xml:"xmlorder"`

	AwsQueryError  []AwsQueryError  `xml:"awsqueryerror"`
	AwsSoap11Error []AwsSoap11Error `xml:"awssoap11error"`
	AwsSoap12Error []AwsSoap12Error `xml:"awssoap12error"`

	BSF []BSF `xml:"bsf"`

	Http       []Http       `xml:"http"`
	HttpHeader []HttpHeader `xml:"httpheader"`
	HttpLabel  []HttpLabel  `xml:"httplabel"`

	Documentation []Documentation `xml:"documentation"`
	Encryption    []Encryption    `xml:"encryption"`
}

type Trait struct {
	Target
}
type Target struct {
	Target string `xml:"target,attr"`
}

func (t Target) Unique() string {
	return t.Target
}

//Traits
type Idempotent struct {
	Trait
}
type Exception struct {
	Trait
}
type Fault struct {
	Trait
}
type Box struct {
	Trait
}
type XmlName struct {
	Trait
	Name Value `xml:"name"`
}

type Value struct {
	Value string `xml:"value,attr"`
}
type XmlNamespace struct {
	Trait
	Namespace Value `xml:"namespace"`
}

type XmlArtificial struct {
	Trait
	Name      Value `xml:"name"`
	Namespace Value `xml:"namespace"`
}
type XmlOrder struct {
	Trait
	Order Value `xml:"order"`
}

//TODO: Java

type AwsError struct {
	Trait
	Code             Value `xml:"code"`
	HttpResponseCode Value `xml:"httpresponsecode"`
	SenderFault      Value `xml:"senderfault"`
}
type AwsQueryError struct {
	AwsError
}
type AwsSoap11Error struct {
	AwsError
}
type AwsSoap12Error struct {
	AwsError
}
type BSF struct {
	Trait
	ClassName Value `xml:"classname"`
}

type Http struct {
	Trait
	Verb Value `xml:"verb"`
	URI  Value `xml:"uri"`
}
type HttpHeader struct {
	Trait
	Header Value `xml:"header"`
}

type HttpLabel struct {
	Trait
	Label Value `xml:"label"`
}
type Documentation struct {
	Trait
	Text string `xml:",chardata"`
}
type Encryption struct {
	Trait
	Algorithm Value `xml:"algorithm"`
}

//TODO: Codigo

//Constraint Traits
type Length struct {
	Trait
	Min Value `xml:"min"`
	Max Value `xml:"max"`
}

type Required struct {
	Trait
}

type Enum struct {
	Trait
	EnumValue []struct {
		Value Value `xml:"value"`
		Name  Value `xml:"name"`
	} `xml:"enumValue"`
}

type Pattern struct {
	Trait
	Regex Value `xml:"regex"`
}

type Range struct {
	Trait
	Min Value `xml:"min"`
	Max Value `xml:"max"`
}

//The standard attributes of a shape
type Shape struct {
	Name string `xml:"name,attr"`
}

func (s Shape) Unique() string {
	return s.Name
}

//Complex Shapes
type Structure struct {
	Shape
	IsA      string            `xml:"isa,attr"`
	Abstract bool              `xml:"abstract,attr"`
	Member   []StructureMember `xml:"member"`
}
type StructureMember struct {
	Shape
	Target
}

type List struct {
	Shape
	Member Target `xml:"member"`
}

type Map struct {
	Shape
	Key   Target `xml:"key"`
	Value Target `xml:"value"`
}

//Service Shapes
type Operation struct {
	Shape
	Input  Target   `xml:"input"`
	Output Target   `xml:"output"`
	Error  []Target `xml:"error"`
}

type Service struct {
	Shape
	Operation []Target `xml:"operation"`
}

//Simple Shapes
type Blob struct {
	Shape
}

type Boolean struct {
	Shape
}

type Byte struct {
	Shape
}

type Timestamp struct {
	Shape
}

type Character struct {
	Shape
}

type Double struct {
	Shape
}
type Float struct {
	Shape
}
type Integer struct {
	Shape
}
type Long struct {
	Shape
}
type Short struct {
	Shape
}
type String struct {
	Shape
	Encoding string `xml:"encoding,attr"`
}

type BigInteger struct {
	Shape
}
type BigDecimal struct {
	Shape
}
type Envelope struct {
	Shape
}
