/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package awsjsonv1

import (
	"CoralGoCodec/codec"
	"CoralRPCGoSupport/internal/roundtrip"
	"CoralRPCGoSupport/rpc"
	"aaa"
	"authv4"
	"authv4/arps"

	"github.com/pkg/errors"
)

// Assert that we fulfill the interfaces
var _ codec.Codec = AWSJSONv1{}
var _ codec.RoundTripper = AWSJSONv1{}
var _ codec.Server = AWSJSONv1{}
var _ roundtrip.Codec = AWSJSONv1{}

// https://w.amazon.com/index.php/Coral/Manual/Protocols#AWS.2FJSON_1.0
type AWSJSONv1 struct {
	host               string                      // required for clients. host or host:port
	path               string                      // optional
	requestIdGenerator rpc.RequestIdGenerator      // optional
	aaaSupport         aaa.Support                 // optional
	aaaClient          aaa.Client                  // optional
	arpsAuthorizer     *arps.ARPSAuthorizer        // optional
	basicAuth          *rpc.BasicAuth              // optional
	bearerTokenVendor  roundtrip.BearerTokenVendor // optional
	securityToken      string                      // optional, see https://w.amazon.com/index.php/Coral/Specifications/HttpSecurityToken
	signerV4           *authv4.Signer              // optional
	tripper            roundtrip.Tripper           // created by New()
}

// hasAuthSet returns true if any of the mechanisms for authenticating/authorizing a
// request/response has been set.
func (c AWSJSONv1) hasAuthSet() bool {
	return c.signerV4 != nil ||
		c.arpsAuthorizer != nil ||
		c.basicAuth != nil ||
		c.bearerTokenVendor != nil ||
		c.securityToken != "" ||
		c.aaaClient != nil ||
		c.aaaSupport != nil
}

// Represents a construction option.
type Option func(*AWSJSONv1) error

// New creates a new instance of AWSJSONv1 that can be used for client or
// server use-cases.  Creates the infrastructure necessary to perform round
// trip requests.
// Example: v1, err := awsjsonv1.New(awsjsonv1.Host("foo.com:8080"), awsjsonv1.WithSignerV4(signer)).
func New(options ...Option) (AWSJSONv1, error) {
	v1 := AWSJSONv1{}

	for _, option := range options {
		if option != nil {
			if err := option(&v1); err != nil {
				return v1, err
			}
		}
	}

	v1.tripper = roundtrip.Tripper{
		Codec:              v1,
		Host:               v1.host,
		RequestIdGenerator: v1.requestIdGenerator,
		AAAClient:          v1.aaaClient,
		ARPSAuthorizer:     v1.arpsAuthorizer,
		BasicAuth:          v1.basicAuth,
		BearerTokenVendor:  v1.bearerTokenVendor,
		SecurityToken:      v1.securityToken,
		SignerV4:           v1.signerV4,
	}

	return v1, nil
}

// WithHost is a client-side parameter for setting what host or
// host:port to make requests against when performing a RoundTrip.
func WithHost(host string) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		v1.host = host
		return nil
	})
}

// WithPath is an optional client-side parameter for setting the
// path, e.g. /bsf, that requests should be send against when performing
// a RoundTrip.
func WithPath(path string) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		v1.path = path
		return nil
	})
}

// WithRequestIdGenerator is an option function that will be used for generating client request ids.
// Amazon standard is to use a UUID v4 identifier for request ids.
func WithRequestIdGenerator(generator rpc.RequestIdGenerator) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		v1.requestIdGenerator = generator
		return nil
	})
}

// WithAAAForClient is an option function to set a AWSJSONv1 AAA client.  Use this version
// when you only need client support.  This can be set to either the Roadside Assist or
// the Security Daemon client. Cannot be used with other Auth options.
func WithAAAForClient(aaa aaa.Client) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		if v1.hasAuthSet() {
			return errors.Errorf("auth has already been set: %#v", v1)
		}
		v1.aaaClient = aaa
		return nil
	})
}

// WithAAAForServer is an option function to set full AAA support.  Use this version
// when you need Server support.  This can only be set to the Security Daemon client.
// Cannot be used with other Auth options.
func WithAAAForServer(aaa aaa.Support) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		if v1.hasAuthSet() {
			return errors.Errorf("auth has already been set: %#v", v1)
		}
		v1.aaaSupport = aaa
		return nil
	})
}

// WithARPSAuthorizer specifies the authorizer to use for authorizing requests.
// Cannot be used with other Auth options.
func WithARPSAuthorizer(authorizer *arps.ARPSAuthorizer) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		if v1.hasAuthSet() {
			return errors.Errorf("auth has already been set: %#v", v1)
		}
		v1.arpsAuthorizer = authorizer
		return nil
	})
}

// WithBasicAuth sets a username + password to set as a basic auth header.
// Cannot be used with other Auth options.
func WithBasicAuth(ba *rpc.BasicAuth) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		if v1.hasAuthSet() {
			return errors.Errorf("auth has already been set: %#v", v1)
		}
		v1.basicAuth = ba
		return nil
	})
}

// WithBearerTokenVendor sets Bearer tokens in HTTP requests. CloudAuth
// for example requires Bearer token to authenticate a client.
func WithBearerTokenVendor(bearerTokenVendor roundtrip.BearerTokenVendor) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		if v1.hasAuthSet() {
			return errors.Errorf("auth has already been set: %#v", v1)
		}
		v1.bearerTokenVendor = bearerTokenVendor
		return nil
	})
}

// WithSecurityToken sets a security token to set as an Authorization header.
// Cannot be used with other Auth options.
func WithSecurityToken(securityToken string) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		if v1.hasAuthSet() {
			return errors.Errorf("auth has already been set: %#v", v1)
		}
		v1.securityToken = securityToken
		return nil
	})
}

// WithSignerV4 sets the AuthV4 Signer to use for signing requests.
// Cannot be used with other Auth options.
func WithSignerV4(signer *authv4.Signer) Option {
	return (Option)(func(v1 *AWSJSONv1) error {
		if v1.hasAuthSet() {
			return errors.Errorf("auth has already been set: %#v", v1)
		}
		v1.signerV4 = signer
		return nil
	})
}
