/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package awsjsonv1

import (
	"CoralRPCGoSupport/internal/test/fake"
	"CoralRPCGoSupport/rpc"
	"testing"
)

func TestNew(t *testing.T) {
	// Make sure that we can construct without any options without causing a panic.
	if _, err := New(); err != nil {
		t.Error("Failure to construct without options:", err)
	}

	host := "host"
	if r, err := New(WithHost(host)); err != nil || r.host != host || r.tripper.Host != host {
		t.Error("Failure to set host or non-nil error:", err)
	}

	path := "/path"
	if r, err := New(WithPath(path)); err != nil || r.path != path {
		t.Error("Failure to set path or non-nil error:", err)
	}

	gen := func() (string, error) { return "1", nil }
	if r, err := New(WithRequestIdGenerator(gen)); err != nil || r.requestIdGenerator == nil || r.tripper.RequestIdGenerator == nil {
		t.Error("Failure to set RequestIdGenerator or non-nil error:", err)
	}

	if r, err := New(WithAAAForServer(&fake.AAA{})); err != nil || r.aaaSupport == nil {
		t.Error("Failure to set AAA client or non-nil error:", err)
	}

	if r, err := New(WithARPSAuthorizer(&fake.ARPSAuthorizer)); err != nil ||
		r.arpsAuthorizer != &fake.ARPSAuthorizer || r.tripper.ARPSAuthorizer != &fake.ARPSAuthorizer {
		t.Error("Failure to set ARPSAuthorizer or non-nil error:", err)
	}

	fakeBearerTokenVendor := &fake.BearerTokenVendor{}
	if r, err := New(WithBearerTokenVendor(fakeBearerTokenVendor)); err != nil ||
		r.bearerTokenVendor != fakeBearerTokenVendor || r.tripper.BearerTokenVendor != fakeBearerTokenVendor {

		t.Error("Failure to set BearerTokenVendor")
	}

	basicAuth := &rpc.BasicAuth{}
	if r, err := New(WithBasicAuth(basicAuth)); err != nil || r.basicAuth != basicAuth || r.tripper.BasicAuth != basicAuth {
		t.Error("Failure to set BasicAuth or non-nil error:", err)
	}

	token := "token"
	if r, err := New(WithSecurityToken(token)); err != nil || r.securityToken != token || r.tripper.SecurityToken != token {
		t.Error("Failure to set security token or non-nil error:", err)
	}

	if r, err := New(WithSignerV4(fake.Signer)); err != nil || r.signerV4 != fake.Signer || r.tripper.SignerV4 != fake.Signer {
		t.Error("Failure to set SignerV4 or non-nil error:", err)
	}
}

func TestNew_WithAuthSet(t *testing.T) {
	optAAAClient := WithAAAForClient(&fake.AAA{})
	optAAAServer := WithAAAForServer(&fake.AAA{})
	optARPS := WithARPSAuthorizer(&fake.ARPSAuthorizer)
	optBearer := WithBearerTokenVendor(&fake.BearerTokenVendor{})
	optBasic := WithBasicAuth(&rpc.BasicAuth{})
	optSigner := WithSignerV4(fake.Signer)
	optToken := WithSecurityToken("token")
	options := []Option{optAAAClient, optAAAServer, optARPS, optBasic, optBearer, optSigner, optToken}

	// Having two auth option functions set auth parameters is not allowed.
	for _, opt1 := range options {
		for _, opt2 := range options {
			if r, err := New(opt1, opt2); err == nil {
				t.Errorf("Expected error but found none for %#v", r)
			}
		}
	}
}
