/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package awsjsonv1

import (
	"CoralGoCodec/codec"
	"CoralGoModel/model"
	"CoralRPCGoSupport/internal/roundtrip"
	cjson "CoralRPCGoSupport/rpc/encoding/json"
	"GoLog/log"
	"aaa"
	"io"
	"io/ioutil"
	"net/http"
	"reflect"
	"strings"

	"github.com/pkg/errors"
)

const (
	headerAmznDate    = "X-Amz-Date"
	headerAmznTarget  = "X-Amz-Target"
	headerContentType = "Content-Type"

	contentType = "application/x-amz-json-1.0"
)

// Marshal supports codec.Codec.
func (c AWSJSONv1) Marshal(obj interface{}) ([]byte, error) {
	return cjson.Marshal(obj)
}

// Unmarshal supports codec.Codec.
func (c AWSJSONv1) Unmarshal(d []byte, obj interface{}) error {
	return cjson.Unmarshal(d, obj, "")
}

// UnmarshalWithService unmarshals the given data into the given obj
// using assemblies from the given service.
func (c AWSJSONv1) UnmarshalWithService(m map[string]interface{}, obj interface{}, service string) error {
	return cjson.UnmarshalMap(m, obj, service)
}

// MarshalRequest satisfies roundtrip.Codec.
func (c AWSJSONv1) MarshalRequest(r *codec.Request) ([]byte, error) {
	return c.Marshal(r.Input)
}

// UnmarshalResponse satisfies roundtrip.Codec.
func (c AWSJSONv1) UnmarshalResponse(rtCtx roundtrip.Context, respBody []byte, obj interface{}, service string) error {
	m, err := cjson.ToMap(respBody)
	if err != nil {
		return roundtrip.NewError("failed to convert response to a map", err, rtCtx)
	}

	// Return the raw error since the error might be from the model.
	return c.UnmarshalWithService(m, obj, service)
}

// IsSupported supports codec.Codec.
// POST http://0.0.0.0:8000/
//
// HTTP/1.1
// Content-Type: application/x-amz-json-1.0
// X-Amz-Date: Thu, 01 Mar 2012 22:07:13 GMT
// X-Amz-Target: WeatherService.GetWeather
//
// {"__type": "GetWeatherInput", "location": "foo"}
func (c AWSJSONv1) IsSupported(g codec.Getter) bool {
	if !strings.Contains(g.Get(headerContentType), contentType) {
		log.Trace("AWSJSONv1: Content type doesn't match", g.Get(headerContentType), contentType)
		return false
	}

	// We don't care about these values, but we care that they are present.
	service, op := GetServiceAndOp(g.Get(headerAmznTarget))
	if g.Get(headerAmznDate) == "" || service == "" || op == "" {
		log.Trace("AWSJSONv1: Missing date, or unable to determine service and op")
		return false
	}

	return true
}

// GetServiceAndOp splits the given val based on the location of the last period character.
// If there is no period character, then empty strings are returned.
func GetServiceAndOp(val string) (service, operation string) {
	if i := strings.LastIndex(val, "."); i > 0 && i < len(val)-1 {
		service, operation = val[:i], val[i+1:]
		return service, operation
	}
	return "", ""
}

// UnmarshalRequest satisfies codec.Server.
func (c AWSJSONv1) UnmarshalRequest(r *http.Request) (*codec.Request, error) {
	var sctx *aaa.ServiceContext
	var err error
	if c.aaaSupport != nil {
		if sctx, err = c.aaaSupport.DecodeRequest(r); err != nil {
			return nil, err
		}
		log.Tracef("Decoded ServiceContext %#v", *sctx)
	}

	if c.arpsAuthorizer != nil {
		auth, err := c.arpsAuthorizer.Authenticate(r)
		if err != nil {
			return nil, errors.Errorf("request is not authorized. error message is %v", err)
		}
		log.Tracef("Request from account %s is authorized", auth.AccountId())
	}

	serviceName, opName := GetServiceAndOp(r.Header.Get(headerAmznTarget))
	if serviceName == "" || opName == "" {
		return nil, errors.New("Request is not supported by the AWSJSONv1 codec")
	}

	defer r.Body.Close()
	reqBody, err := ioutil.ReadAll(r.Body)
	if err != nil {
		return nil, err
	}

	svc := model.LookupService(serviceName)
	var asmName string
	// Find the assembly that has a Service that matches our serviceName
	for _, assembly := range svc.Assemblies() {
		for _, op := range assembly.Ops() {
			if op.Name() == opName {
				asmName = assembly.Name()
				break
			}
		}
	}
	asm := svc.Assembly(asmName)
	op, err := asm.Op(opName)
	if err != nil {
		return nil, errors.Wrap(err, "Unable to retrieve operation "+opName)
	}

	// Make sure that the auth context has the service and operation name from the request.
	if sctx != nil && sctx.Service == "" {
		sctx.Service = serviceName
		sctx.Operation = opName
	}

	// Now that we have all of the information we need to authorize the request, perform
	// authorization before dispatching to the service.
	if c.aaaSupport != nil {
		auth, err := c.aaaSupport.AuthorizeRequest(sctx)
		if err != nil {
			return nil, err
		}
		if !auth.Authorized {
			return nil, errors.New("Request is not authorized.  Error message is " + auth.ErrorMessage)
		}
		log.Trace("Request is authorized with code", auth.AuthorizationCode)
	}

	// Create the base codec.Request, then create instances of the input
	// and output if they are defined for the operation.
	cr := &codec.Request{
		Service:   codec.ShapeRef{AsmName: asmName, ShapeName: serviceName},
		Operation: codec.ShapeRef{AsmName: asmName, ShapeName: opName},
		AuthCtx:   sctx,
	}
	if input := op.Input(); input != nil {
		m, mapErr := cjson.ToMap(reqBody)
		if mapErr != nil {
			return nil, mapErr
		}
		targetType := reflect.TypeOf(input.New()).Elem()
		shape := cjson.DetermineShape(reflect.ValueOf(m), targetType, cr.Service.ShapeName)
		cr.Input = shape.New()
		if err := c.UnmarshalWithService(m, cr.Input, cr.Service.ShapeName); err != nil {
			return nil, err
		}
	}
	if output := op.Output(); output != nil {
		cr.Output = output.New()
	}
	return cr, nil
}

// MarshalResponse satisfies codec.Server.
func (c AWSJSONv1) MarshalResponse(w http.ResponseWriter, r *codec.Request) {
	if r == nil || r.Operation.ShapeName == "" || r.Service.ShapeName == "" {
		log.Fatal("MarshalResponse called without proper input.")
		http.Error(w, "Unable to process request", http.StatusInternalServerError)
		return
	}

	var body []byte
	var err error

	// Serialize r.Output into bytes if available.
	if r.Output != nil {
		body, err = c.Marshal(r.Output)
		if err != nil {
			http.Error(w, "Unable to process response", http.StatusInternalServerError)
			return
		}
	}

	// Set the appropriate response headers for the codec.
	headers := w.Header()
	headers.Set(headerContentType, contentType)

	if sctx, ok := r.AuthCtx.(*aaa.ServiceContext); ok && c.aaaSupport != nil {
		body, err = c.aaaSupport.EncodeResponse(sctx, headers, body)
		if err != nil {
			log.Fatalf("Error encoding response using AAA: %+v", err)
			http.Error(w, "Unable to encode response", http.StatusInternalServerError)
			return
		}
		log.Trace("Output has been AAA encoded")
	}

	// Send the response.
	if body != nil {
		w.Write(body)
	} else {
		w.WriteHeader(http.StatusNoContent)
	}
}

// RoundTrip supports codec.RoundTripper.
func (c AWSJSONv1) RoundTrip(r *codec.Request, rw io.ReadWriter) error {
	if r.RequestHeaders == nil {
		r.RequestHeaders = make(map[string]string)
	}
	r.RequestHeaders[headerContentType] = contentType
	r.RequestHeaders[headerAmznTarget] = r.Service.ShapeName + "." + r.Operation.ShapeName
	return c.tripper.Go(c.path, r, rw)
}
