/* Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.*/

// Package fake provides dummy structs and interface implementations for things like AAA or AWS Signers
package fake

import (
	"aaa"
	"authv4"
	"authv4/arps"
	"bytes"
	"context"
	"net/http"
	"time"

	"github.com/pkg/errors"
	"golang.a2z.com/cloudauth"
)

const Region = "us-east-1"
const Service = "fakeService"
const AccessKey = "none"
const SecretKey = "none"

var (
	// Signer is an initialized authv4 signer
	Signer         = authv4.NewSigner(Region, Service, AccessKey, SecretKey, []string{AccessKey})
	ARPSAuthorizer = arps.ARPSAuthorizer{Authorizer: authv4.NewAuthorizer(&Authenticator{false, Signer}, []string{AccessKey})}
)

// AAA provides a fake implementation of aaa.Support.
type AAA struct {
	AuthResult *aaa.AuthorizationResult
	AuthErr    error
	EncodeErr  error
	DecodeErr  error
}

// SanityCheck satisfies aaa.Support.
func (f *AAA) SanityCheck() error { return nil }

// EncodeRequest satisfies aaa.Support.
func (f *AAA) EncodeRequest(service, operation string, r *http.Request) (*aaa.ClientContext, error) {
	if f.EncodeErr == nil {
		return &aaa.ClientContext{}, nil
	}
	return nil, f.EncodeErr
}

// DecodeResponse satisfies aaa.Support.
func (f *AAA) DecodeResponse(clientCxt *aaa.ClientContext, resp *http.Response) error {
	return f.DecodeErr
}

// DecodeRequest satisfies aaa.Support.
func (f *AAA) DecodeRequest(r *http.Request) (*aaa.ServiceContext, error) {
	if f.DecodeErr == nil {
		return &aaa.ServiceContext{}, nil
	}
	return nil, f.DecodeErr
}

// AuthorizeRequest satisfies aaa.Support.
func (f *AAA) AuthorizeRequest(ctx *aaa.ServiceContext) (*aaa.AuthorizationResult, error) {
	return f.AuthResult, f.AuthErr
}

// EncodeResponse satisfies aaa.Support.
func (f *AAA) EncodeResponse(ctx *aaa.ServiceContext, h http.Header, body []byte) ([]byte, error) {
	return body, f.EncodeErr
}

// BearerTokenVendor provides a fake implementation of roundtrip.BearerTokenVendor.
type BearerTokenVendor struct {
	token string
	err   error
}

// Vend satisfies BearerTokenVendor.
func (b *BearerTokenVendor) Vend(hostname string) (string, error) {
	return b.token, b.err
}

// CloudAuth provides a dummy implementation of CloudAuth
type CloudAuth struct {
	Err       error
	Service   string
	Operation string
}

func (f *CloudAuth) Introspect(accessToken string) (*cloudauth.IntrospectResponse, error) {
	return &cloudauth.IntrospectResponse{}, nil
}

func (f *CloudAuth) WithClient(ctx context.Context) context.Context {
	return ctx
}

func (f *CloudAuth) TokenEndpoint() string {
	return "http://cloudauth.com"
}

func (f *CloudAuth) AuthorizeRequest(r *http.Request, svcName string, opName string) (*cloudauth.AuthorizationResult, error) {
	if f.Err != nil {
		return nil, f.Err
	}
	if f.Service == svcName && f.Operation == opName {
		return &cloudauth.AuthorizationResult{Result: cloudauth.ResultAllow}, nil
	} else {
		return &cloudauth.AuthorizationResult{Result: cloudauth.ResultDeny, BearerChallenge: "challenge"}, nil
	}
}

// Pipe satisfies io.ReadWriter
type Pipe struct {
	ReadBuf  *bytes.Buffer
	WriteBuf bytes.Buffer
}

// Read satisfies io.Reader.
func (f *Pipe) Read(p []byte) (n int, err error) {
	return f.ReadBuf.Read(p)
}

// Write satisfies io.Writer.
func (f *Pipe) Write(p []byte) (n int, err error) {
	return f.WriteBuf.Write(p)
}

type Authenticator struct {
	alwaysFail bool
	signer     *authv4.Signer
}

func (f *Authenticator) Authenticate(claim *authv4.AuthV4Claim) (*authv4.AuthenticatedClient, error) {
	if f.alwaysFail {
		return nil, errors.New("Failed")
	}
	_, nowDay := authv4.NewSigningDay(time.Now())
	return authv4.NewAuthenticatedClient(claim.AccessKey, AccessKey, f.signer.GetSigningKey(nowDay, SecretKey)), nil
}

func (f *Authenticator) SetDebugLogger(debugLogger authv4.DebugLogger) {}
func (f *Authenticator) DebugLoggerEnable()                            {}
func (f *Authenticator) DebugLoggerDisable()                           {}
func (f *Authenticator) DebugLoggerIsEnabled() bool {
	return true
}
