// Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.

// Package fakemodel provides fake Coral models for testing.
package fakemodel

import (
	"CoralGoCodec/codec"
	"CoralGoModel/model"
	"reflect"
)

const (
	serviceName          = "TopListsService"
	serviceName2         = "OtherService"
	topListsAssemblyName = "com.amazon.toplistsservice"
	otherAssemblyName    = "rpcv1_test.other"
	rpcv1AssemblyName    = "rpcv1_test"
	rpcv1ExtAssemblyName = "rpcv1_test.ext"
)

func init() {
	// Register our fake shapes
	var val1 FakeInput
	t := reflect.TypeOf(&val1)
	asm := model.LookupService(serviceName).Assembly(rpcv1AssemblyName)
	asm.RegisterShape("FakeInput", t, func() interface{} {
		return NewFakeInput()
	})
	var val2 FakeRule
	t = reflect.TypeOf(&val2)
	asm.RegisterShape("FakeRule", t, func() interface{} {
		return NewFakeRule()
	})
	var val3 FakeOutput
	t = reflect.TypeOf(&val3)
	asm.RegisterShape("FakeOutput", t, func() interface{} {
		return NewFakeOutput()
	})
	var val4 FakeOutputExtension
	t = reflect.TypeOf(&val4)
	asm.RegisterShape("FakeOutputExtension", t, func() interface{} {
		return NewFakeOutputExtension()
	})
	fakeInput, _ := asm.Shape("FakeInput")
	fakeOutput, _ := asm.Shape("FakeOutput")
	asm.RegisterOp("IsAuthorized", fakeInput, fakeOutput, nil)

	// Register our extension to FakeOutput so that we can test unmarshalling
	// an extension to a base type.
	var tloExt FakeOutputExtension
	t = reflect.TypeOf(&tloExt)
	model.LookupService(serviceName).Assembly(rpcv1ExtAssemblyName).RegisterShape("FakeOutputExtension", t, func() interface{} {
		return NewFakeOutputExtension()
	})

	// Register our shape from another service so that we can test unmarshalling
	// concrete types from another service
	var other FakeOther
	t = reflect.TypeOf(&other)
	model.LookupService(serviceName2).Assembly(otherAssemblyName).RegisterShape("FakeOther", t, func() interface{} {
		return NewFakeOther()
	})

	// Register FindTopListInput with Coral
	var val FindTopListInput
	t = reflect.TypeOf(&val)
	model.LookupService(serviceName).Assembly(topListsAssemblyName).RegisterShape("FindTopListInput", t, func() interface{} {
		return NewFindTopListInput()
	})

	// Register FindTopListOutput with Coral
	var tlo FindTopListOutput
	t = reflect.TypeOf(&tlo)
	model.LookupService(serviceName).Assembly(topListsAssemblyName).RegisterShape("FindTopListOutput", t, func() interface{} {
		return NewFindTopListOutput()
	})

	// Register FindTopListInput and FindTopListOutput with the FindTopList operation
	asm = model.LookupService(serviceName).Assembly(topListsAssemblyName)
	input, _ := asm.Shape("FindTopListInput")
	output, _ := asm.Shape("FindTopListOutput")
	asm.RegisterOp("FindTopList", input, output, nil)
}

var (
	// TestService is a stub shape ref for serviceName
	TestService = codec.ShapeRef{
		AsmName:   "rpcv1_test",
		ShapeName: serviceName,
	}
	// TestOperation is a stub shape ref for serviceName.IsAuthorized
	TestOperation = codec.ShapeRef{
		AsmName:   "rpcv1_test",
		ShapeName: "IsAuthorized",
	}
)

// FakeInput is a dummy implementation of a Coral input model
type FakeInput interface {
	SetUser(v string)
	SetRules(v []FakeRule)
	SetOther(v FakeOther)
	User() string
	Rules() []FakeRule
	Other() FakeOther
}

type _FakeInput struct {
	I_user  string     `coral:"user"`
	I_rules []FakeRule `coral:"rules"`
	I_other FakeOther  `coral:"other"`
}

func (f *_FakeInput) SetUser(u string) {
	f.I_user = u
}

func (f *_FakeInput) SetRules(r []FakeRule) {
	f.I_rules = r
}

func (f *_FakeInput) SetOther(o FakeOther) {
	f.I_other = o
}

func (f *_FakeInput) User() string {
	return f.I_user
}

func (f *_FakeInput) Rules() []FakeRule {
	return f.I_rules
}

func (f *_FakeInput) Other() FakeOther {
	return f.I_other
}

func NewFakeInput() FakeInput {
	return &_FakeInput{}
}

// FakeRule is a dummy implementation of a Coral model
type FakeRule interface {
	SetSource(string)
	SetIdentifier(string)
}

type _FakeRule struct {
	I_source     string `coral:"source"`
	I_identifier string `coral:"identifier"`
}

func (rule *_FakeRule) SetSource(s string) {
	rule.I_source = s
}

func (rule *_FakeRule) SetIdentifier(i string) {
	rule.I_identifier = i
}

func (rule *_FakeRule) Source() string {
	return rule.I_source
}

func (rule *_FakeRule) Identifier() string {
	return rule.I_identifier
}

func NewFakeRule() FakeRule {
	return &_FakeRule{}
}

// FakeOther is a dummy implementation of a Coral model
type FakeOther interface {
	SetSource(string)
	SetIdentifier(string)
}

type _FakeOther struct {
	I_source     string `coral:"source"`
	I_identifier string `coral:"identifier"`
}

func (Other *_FakeOther) SetSource(s string) {
	Other.I_source = s
}

func (Other *_FakeOther) SetIdentifier(i string) {
	Other.I_identifier = i
}

func (Other *_FakeOther) Source() string {
	return Other.I_source
}

func (Other *_FakeOther) Identifier() string {
	return Other.I_identifier
}

func NewFakeOther() FakeOther {
	return &_FakeOther{}
}

// FakeOutput is a dummy implementation of a Coral Output model
type FakeOutput interface {
	SetOutput(v string)
	Output() string
}

type _FakeOutput struct {
	I_output string `coral:"output"`
}

func (f *_FakeOutput) SetOutput(s string) {
	f.I_output = s
}

func (f *_FakeOutput) Output() string {
	return f.I_output
}

func NewFakeOutput() FakeOutput {
	return &_FakeOutput{}
}

// FakeOutputExtension is a dummy extension of FakeOutput.
type FakeOutputExtension interface {
	SetOutput(v string)
	Output() string
	SetExt(FakeOutput)
	Ext() FakeOutput
}

type _FakeOutputExtension struct {
	I_output string     `coral:"output"`
	I_ext    FakeOutput `coral:"ext"`
}

func (ext *_FakeOutputExtension) SetOutput(s string) {
	ext.I_output = s
}

func (ext *_FakeOutputExtension) SetExt(i FakeOutput) {
	ext.I_ext = i
}

func (ext *_FakeOutputExtension) Output() string {
	return ext.I_output
}

func (ext *_FakeOutputExtension) Ext() FakeOutput {
	return ext.I_ext
}

func NewFakeOutputExtension() FakeOutputExtension {
	return &_FakeOutputExtension{}
}

type FindTopListInput interface {
	__type()
	Name() *string
	SetName(v *string)
}
type _FindTopListInput struct {
	Ị_name *string `coral:"name"`
}

func (input *_FindTopListInput) Name() *string {
	return input.Ị_name
}
func (input *_FindTopListInput) SetName(v *string) {
	input.Ị_name = v
}
func (input *_FindTopListInput) __type() {
}
func NewFindTopListInput() FindTopListInput {
	return &_FindTopListInput{}
}

// Test Output
type FindTopListOutput interface {
	__type()
	Name() *string
	SetName(v *string)
}
type _FindTopListOutput struct {
	Ị_name *string `coral:"name"`
}

func (output *_FindTopListOutput) Name() *string {
	return output.Ị_name
}
func (output *_FindTopListOutput) SetName(v *string) {
	output.Ị_name = v
}
func (output *_FindTopListOutput) __type() {
}
func NewFindTopListOutput() FindTopListOutput {
	return &_FindTopListOutput{}
}

// FakeEmptyOutput is a model which is "empty".
type FakeEmptyOutput interface{}

type _FakeEmptyOutput struct{}

func NewFakeEmptyOutput() FakeEmptyOutput {
	return &_FakeEmptyOutput{}
}

// FakeErrorOutput satisfies the error interface.
type FakeErrorOutput interface {
	error
	SetMessage(v string)
	Message() string
}

type _FakeErrorOutput struct {
	Ị_message string `coral:"message"`
}

func (output *_FakeErrorOutput) Error() string {
	return model.ErrorMessage(output)
}
func (output *_FakeErrorOutput) Message() string {
	return output.Ị_message
}
func (output *_FakeErrorOutput) SetMessage(v string) {
	output.Ị_message = v
}

func NewFakeErrorOutput() FakeErrorOutput {
	return &_FakeErrorOutput{}
}

// FakeOutputLooksLikeError is an output that looks exactly like an error.
type FakeOutputLooksLikeError interface {
	error
	SetMessage(v string)
	Message() string
}

type _FakeOutputLooksLikeError struct {
	Ị_message string `coral:"message"`
}

func (output *_FakeOutputLooksLikeError) Error() string {
	return model.ErrorMessage(output)
}
func (output *_FakeOutputLooksLikeError) Message() string {
	return output.Ị_message
}
func (output *_FakeOutputLooksLikeError) SetMessage(v string) {
	output.Ị_message = v
}
func NewFakeOutputLooksLikeError() FakeOutputLooksLikeError {
	return &_FakeOutputLooksLikeError{}
}
