// Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.

package json

import (
	"CoralGoModel/model"
	"CoralRPCGoSupport/internal/test/fakemodel"
	"math/big"
	"reflect"
	"strings"
	"testing"
	"time"

	"github.com/pkg/errors"
)

const (
	testService = "testService"
)

type encodeTest struct {
	Foo *string     `coral:"foo"`
	Bar *string     `coral:"baz"`
	Baz *encodeTest `coral:"bar"`
	Qux *big.Rat    `coral:"qux"`
}

func TestEncode(t *testing.T) {
	h, w := "Hello", "World"
	c, g := "Coral", "Go"
	r := new(big.Rat).SetFloat64(0.6)

	val := reflect.ValueOf(&encodeTest{&h, &w, &encodeTest{&c, &g, nil, nil}, r})
	m, ok := encode(val).(map[string]interface{})
	if !ok {
		t.Fatalf("Invalid type received from encode, want map[string]interface{}, got %T", encode(val))
	}
	if m["foo"] != h {
		t.Errorf(`m["foo"] = "%v", want "%s"`, m["foo"], h)
	}
	if m["baz"] != w {
		t.Errorf(`m["baz"] = %v", want "%s"`, m["baz"], w)
	}
	bigRatAsFloat, _ := r.Float64()
	if m["qux"] != bigRatAsFloat {
		t.Errorf(`m["qux"] = %v",big.Rat %s want "%f"`, m["qux"], r, bigRatAsFloat)
	}
	if baz, ok := m["bar"].(map[string]interface{}); ok {
		if baz["foo"] != c {
			t.Errorf(`baz["foo"] = "%v", want "%s"`, baz["foo"], c)
		}
		if baz["baz"] != g {
			t.Errorf(`baz["baz"] = %v, want "%s"`, baz["baz"], g)
		}
		if baz["bar"] != nil {
			t.Errorf(`baz["bar"] = %v, want nil`, baz["bar"])
		}
	} else {
		t.Errorf(`m["bar"] = %t, want map[string]interface{}`, m["bar"])
	}
}

type timeTest *time.Time

func TestTime(t *testing.T) {
	origTime := time.Now()
	var test = timeTest(&origTime)
	val := reflect.ValueOf(test)
	if f, ok := encode(val).(float64); !ok {
		t.Fatalf("Invalid Type received from encode, want float64, got %T", encode(val))
	} else if int64(f) != origTime.Unix() {
		t.Fatalf("Wrong time received from encode, got %v, want %v", int64(f), origTime.Unix())
	}
	var timeTestVal timeTest = new(time.Time)
	timeTestType := reflect.TypeOf(timeTestVal)
	newVal := decode(reflect.ValueOf(encode(val)), timeTestType, testService).Interface()
	if timeVal, ok := newVal.(timeTest); !ok {
		t.Fatalf("Invalid Type received from decode, want timeTest, got %T", newVal)
	} else if (*time.Time)(timeVal).Unix() != origTime.Unix() {
		t.Fatalf("Wrong time received from decode, got %v, want %v", time.Time(*timeVal).Unix(), origTime.Unix())
	}
}

var testTimes = []struct {
	name   string
	format string
	value  string
	round  time.Duration // Round is required as floats are non-exact and we only need to validate the precision at the same level we pass in
}{
	{"Nanosecond precision 1", "2006-01-02T15:04:05.000000000", "2017-12-18T23:07:04.726999998", time.Nanosecond},
	{"Microsecond precision 1", "2006-01-02T15:04:05.000000", "2017-12-18T23:07:04.726999", time.Microsecond},
	{"Millisecond precision 1", "2006-01-02T15:04:05.000", "2017-12-18T23:07:04.726", time.Millisecond},
	{"Millisecond precision 2", "2006-01-02T15:04:05.000", "2017-12-18T23:07:04.727", time.Millisecond},
}

func TestTimeNs(t *testing.T) {
	for _, testTime := range testTimes {
		testTime := testTime
		t.Run(testTime.name, func(t *testing.T) {
			t.Parallel()
			origTime, _ := time.Parse(testTime.format, testTime.value)
			var test = timeTest(&origTime)
			val := reflect.ValueOf(test)

			var valNs float64

			valNs = float64(origTime.UnixNano())
			valNs /= float64(time.Second)

			if f, ok := encode(val).(float64); !ok {
				t.Fatalf("Invalid Type received from encode, want float64, got %T", encode(val))
			} else if f != valNs {
				t.Fatalf("Wrong time received from encode, got %g, want %g", f, valNs)
			}
			var timeTestVal timeTest = new(time.Time)
			timeTestType := reflect.TypeOf(timeTestVal)
			newVal := decode(reflect.ValueOf(encode(val)), timeTestType, testService).Interface()
			if timeVal, ok := newVal.(timeTest); !ok {
				t.Fatalf("Invalid Type received from decode, want timeTest, got %T", newVal)
			} else if (*time.Time)(timeVal).Round(testTime.round).UnixNano() != origTime.UnixNano() {
				t.Fatalf("Wrong time received from decode,\n got  %v\n want %v", time.Time(*timeVal).UnixNano(), origTime.UnixNano())
			}
		})
	}
}

func TestBool(t *testing.T) {
	boolPtr := reflect.PtrTo(reflect.TypeOf(false))
	val := decode(reflect.ValueOf(true), boolPtr, testService).Interface()
	if res, ok := val.(*bool); !ok {
		t.Fatalf("Invalid Type received from decode, want *bool, got %T", res)
	} else if *res != true {
		t.Fatalf("Invalid value received from decode, want true, got %t", *res)
	}
	for _, v := range []string{"1", "t", "T", "true", "TRUE", "True"} {
		val := decode(reflect.ValueOf(v), boolPtr, testService).Interface()
		if res, ok := val.(*bool); !ok {
			t.Fatalf("Invalid Type received from decode, want *bool, got %T", res)
		} else if *res != true {
			t.Fatalf("Invalid value received from decode, want true, got %t", *res)
		}
	}
	for _, v := range []string{"0", "f", "F", "false", "FALSE", "False"} {
		val := decode(reflect.ValueOf(v), boolPtr, testService).Interface()
		if res, ok := val.(*bool); !ok {
			t.Fatalf("Invalid Type received from decode, want *bool, got %T", res)
		} else if *res != false {
			t.Fatalf("Invalid value received from decode, want false, got %t", *res)
		}
	}
}

func TestBoolPanic(t *testing.T) {
	defer func() {
		if r := recover(); r == nil {
			t.Errorf("decode() is supposed to panic when trying to get boolean from unknown string")
		}
	}()
	decode(reflect.ValueOf("blabla"), reflect.PtrTo(reflect.TypeOf(false)), testService).Interface()
}

func TestBigRat(t *testing.T) {
	var test = 75.5
	val := reflect.ValueOf(test)
	var bigRatType = reflect.TypeOf(new(big.Rat))

	decoded := decode(val, bigRatType, testService)

	if !decoded.Type().AssignableTo(bigRatType) {
		t.Fatalf("Type of: %s is not assignable to Type %s", decoded.Type(), bigRatType)
	}
}

var (
	testByteSlice = []byte{
		137, 234, 174, 218, 85, 248, 147, 213, 166, 116, 72, 120, 235, 252, 116, 35, 117, 115, 150, 200, 21, 63,
	}

	testByteSliceEncoded = "iequ2lX4k9WmdEh46/x0I3VzlsgVPw=="

	testEmptySlice = []interface{}{}
	testNilSlice   []interface{}
)

func TestByteSlice(t *testing.T) {
	val := reflect.ValueOf(testByteSlice)
	encoded := encode(val)
	s, ok := encoded.(string)
	if !ok {
		t.Fatalf("Invalid Type received from encode, want string, got %T", encoded)
	} else if s != testByteSliceEncoded {
		t.Fatal("Unexpected value received from encode, want", testByteSliceEncoded, "got", s)
	}

	decoded := decode(reflect.ValueOf(s), reflect.TypeOf(testByteSlice), testService)
	b, ok := decoded.Interface().([]byte)
	if !ok {
		t.Fatalf("Invalid Type received from decode, want []byte, got %T", decoded)
	} else if !reflect.DeepEqual(b, testByteSlice) {
		t.Fatal("Unexpected value received from decode, want", testByteSlice, "got", b)
	}
}

func TestByteSlicePanic(t *testing.T) {
	defer func() {
		if r := recover(); r == nil {
			t.Errorf("decode() is supposed to panic when trying to get []byte from a string that isn't Base64 encoded")
		}
	}()
	decode(reflect.ValueOf("Bob?"), reflect.TypeOf(testByteSlice), testService)
}

func TestEmptySlice(t *testing.T) {
	val := reflect.ValueOf(testEmptySlice)
	encoded := encode(val)
	s, ok := encoded.([]interface{})
	if !ok {
		t.Fatalf("Invalid Type received from encode, want empty slice, got %T", encoded)
	} else if s == nil || len(s) != 0 {
		t.Fatal("Unexpected value received from encode, want empty slice [] but not get it")
	}

	decoded := decode(reflect.ValueOf(s), reflect.TypeOf(testEmptySlice), testService)
	b, ok := decoded.Interface().([]interface{})
	if !ok {
		t.Fatalf("Invalid Type received from decode, want []interface{}, got %T", decoded)
	} else if !reflect.DeepEqual(b, testEmptySlice) {
		t.Fatal("Unexpected value received from decode, want empty slice [] but did not get it")
	}
}

func TestNilSlice(t *testing.T) {
	val := reflect.ValueOf(testNilSlice)
	encoded := encodeSlice(val)
	if encoded != nil {
		t.Fatalf("Invalid Type received from encode, want <nil>, got %T", encoded)
	}

	// So we have to compare with testEmptySlice, as the current decoder always forces
	// a slice to be non-nil. This prevents us from being able to decode back to the
	// nil slice
	decoded := decode(reflect.ValueOf(testNilSlice), reflect.TypeOf(testNilSlice), testService)
	b, ok := decoded.Interface().([]interface{})
	if !ok {
		t.Fatalf("Invalid Type received from decode, want <nil>, got %T", decoded)
	} else if !reflect.DeepEqual(b, testEmptySlice) {
		t.Fatalf("Unexpected value received from decode, want %T but did not get it", testNilSlice)
	}
}

func TestUnmarshalMap(t *testing.T) {
	tests := map[string]struct {
		register func(t *testing.T, assembly model.Assembly)
		data     map[string]interface{}
		obj      interface{}
		want     func(i interface{}) error
		err      error
	}{
		"panic": {
			register: func(t *testing.T, assembly model.Assembly) {
				var val fakemodel.FakeOutput
				Check(t)(assembly.RegisterShape("FakeOutput", reflect.TypeOf(&val), func() interface{} {
					return fakemodel.NewFakeOutput()
				}))
			},
			data: map[string]interface{}{
				"__type": "UnknownShape",
			},
			obj: func() interface{} {
				var output fakemodel.FakeOutput
				return &output
			}(),
			err: errors.New("Error: error for fqn UnknownShape: No Shape registered with the name UnknownShape"),
			want: func(i interface{}) error {
				if v := *i.(*fakemodel.FakeOutput); v != nil {
					t.Errorf("underlying output mismatch: got: %#v; want: <nil>", v)
				}
				return nil
			},
		},
		"basic": {
			register: func(t *testing.T, assembly model.Assembly) {
				var val fakemodel.FakeOutput
				Check(t)(assembly.RegisterShape("FakeOutput", reflect.TypeOf(&val), func() interface{} {
					return fakemodel.NewFakeOutput()
				}))
			},
			data: map[string]interface{}{
				"__type": "FakeOutput",
				"output": "bar",
			},
			obj: func() interface{} {
				var output fakemodel.FakeOutput
				return &output
			}(),
			want: func(i interface{}) error {
				want := "bar"
				if got := (*i.(*fakemodel.FakeOutput)).Output(); got != want {
					return errors.Errorf("output mismatch: got: %s; want: %s", got, want)
				}
				return nil
			},
		},
		"error": {
			register: func(t *testing.T, assembly model.Assembly) {
				var fakeOutputVal fakemodel.FakeOutput
				Check(t)(assembly.RegisterShape("FakeOutput", reflect.TypeOf(&fakeOutputVal), func() interface{} {
					return fakemodel.NewFakeOutput()
				}))
				var fakeErrorOutputVal fakemodel.FakeErrorOutput
				Check(t)(assembly.RegisterShape("FakeErrorOutput", reflect.TypeOf(&fakeErrorOutputVal), func() interface{} {
					return fakemodel.NewFakeErrorOutput()
				}))
			},
			data: map[string]interface{}{
				"__type":  "FakeErrorOutput",
				"message": "error message",
			},
			obj: func() interface{} {
				var output fakemodel.FakeOutput
				return &output
			}(),
			err: errors.New("*fakemodel._FakeErrorOutput: error message"),
			want: func(i interface{}) error {
				if got := *i.(*fakemodel.FakeOutput); got != nil {
					return errors.Errorf("underlying object is not nil as expected: %v", got)
				}
				return nil
			},
		},
		"basic_no_type_defined": {
			register: func(t *testing.T, assembly model.Assembly) {
				var val fakemodel.FakeOutput
				Check(t)(assembly.RegisterShape("FakeOutput", reflect.TypeOf(&val), func() interface{} {
					return fakemodel.NewFakeOutput()
				}))
			},
			data: map[string]interface{}{
				"output": "bar",
			},
			obj: func() interface{} {
				var output fakemodel.FakeOutput
				return &output
			}(),
			want: func(i interface{}) error {
				want := "bar"
				if got := (*i.(*fakemodel.FakeOutput)).Output(); got != want {
					return errors.Errorf("output mismatch: got: %s; want: %s", got, want)
				}
				return nil
			},
		},
		"coral_base_error": {
			register: func(t *testing.T, assembly model.Assembly) {
				var fakeOutputVal fakemodel.FakeOutput
				Check(t)(assembly.RegisterShape("FakeOutput", reflect.TypeOf(&fakeOutputVal), func() interface{} {
					return fakemodel.NewFakeOutput()
				}))
				var fakeErrorOutputVal fakemodel.FakeErrorOutput
				Check(t)(assembly.RegisterShape("FakeErrorOutput", reflect.TypeOf(&fakeErrorOutputVal), func() interface{} {
					return fakemodel.NewFakeErrorOutput()
				}))
			},
			data: map[string]interface{}{
				"__type":  "com.amazon.coral.service#InternalFailure",
				"message": "error message",
			},
			obj: func() interface{} {
				var output fakemodel.FakeOutput
				return &output
			}(),
			err: errors.New("*com_amazon_coral_service._InternalFailure: error message"),
			want: func(i interface{}) error {
				if got := *i.(*fakemodel.FakeOutput); got != nil {
					return errors.Errorf("underlying object is not nil as expected: %v", got)
				}
				return nil
			},
		},
		"unknown_error": {
			register: func(t *testing.T, assembly model.Assembly) {
				var fakeOutputVal fakemodel.FakeOutput
				Check(t)(assembly.RegisterShape("FakeOutput", reflect.TypeOf(&fakeOutputVal), func() interface{} {
					return fakemodel.NewFakeOutput()
				}))
				var fakeErrorOutputVal fakemodel.FakeErrorOutput
				Check(t)(assembly.RegisterShape("FakeErrorOutput", reflect.TypeOf(&fakeErrorOutputVal), func() interface{} {
					return fakemodel.NewFakeErrorOutput()
				}))
			},
			data: map[string]interface{}{
				"__type":  "something.completely.random#Failure",
				"message": "error message",
			},
			obj: func() interface{} {
				var output fakemodel.FakeOutput
				return &output
			}(),
			err: errors.New("No Shape registered with the name Failure"),
			want: func(i interface{}) error {
				if got := *i.(*fakemodel.FakeOutput); got != nil {
					return errors.Errorf("underlying object is not nil as expected: %v", got)
				}
				return nil
			},
		},
		"error_empty_interface_output": {
			register: func(t *testing.T, assembly model.Assembly) {
				var fakeEmptyOutput fakemodel.FakeEmptyOutput
				Check(t)(assembly.RegisterShape("FakeEmptyOutput", reflect.TypeOf(&fakeEmptyOutput), func() interface{} {
					return fakemodel.NewFakeEmptyOutput()
				}))
				var fakeErrorOutput fakemodel.FakeErrorOutput
				Check(t)(assembly.RegisterShape("FakeErrorOutput", reflect.TypeOf(&fakeErrorOutput), func() interface{} {
					return fakemodel.NewFakeErrorOutput()
				}))
			},
			data: map[string]interface{}{
				"__type":  "FakeErrorOutput",
				"message": "error message",
			},
			obj: func() interface{} {
				var output fakemodel.FakeEmptyOutput
				return &output
			}(),
			err: errors.New("*fakemodel._FakeErrorOutput: error message"),
			want: func(i interface{}) error {
				if got := *i.(*fakemodel.FakeEmptyOutput); got != nil {
					return errors.Errorf("underlying object is not nil as expected: %v", got)
				}
				return nil
			},
		},
		"error_when_output_looks_exactly_like_an_error": {
			register: func(t *testing.T, assembly model.Assembly) {
				var fakeOutputLooksLikeErrorVal fakemodel.FakeOutputLooksLikeError
				Check(t)(assembly.RegisterShape("FakeOutputLooksLikeError", reflect.TypeOf(&fakeOutputLooksLikeErrorVal), func() interface{} {
					return fakemodel.NewFakeOutputLooksLikeError()
				}))
				var fakeErrorOutputVal fakemodel.FakeErrorOutput
				Check(t)(assembly.RegisterShape("FakeErrorOutput", reflect.TypeOf(&fakeErrorOutputVal), func() interface{} {
					return fakemodel.NewFakeErrorOutput()
				}))
			},
			data: map[string]interface{}{
				"__type":  "FakeErrorOutput",
				"message": "error message",
			},
			obj: func() interface{} {
				var output fakemodel.FakeOutputLooksLikeError
				return &output
			}(),
			err: nil,
			want: func(i interface{}) error {
				feo := fakemodel.NewFakeErrorOutput()
				reflect.ValueOf(feo).Elem().Set(reflect.ValueOf(*i.(*fakemodel.FakeOutputLooksLikeError)).Elem())

				want := "error message"
				if got := feo.Message(); got != want {
					return errors.Errorf("output mismatch: got: %s; want: %s", got, want)
				}
				return nil
			},
		},
	}

	for name, test := range tests {
		name, test := name, test
		t.Run(name, func(t *testing.T) {
			if test.register != nil {
				test.register(t, model.LookupService(name).Assembly("TestAssembly"))
			}

			gotErr := UnmarshalMap(test.data, test.obj, name)
			if !ErrorContains(gotErr, test.err) {
				t.Fatalf("error value mismatch: got: %s, want it to contain: %s", ErrorValue(gotErr), ErrorValue(test.err))
			}

			if err := test.want(test.obj); err != nil {
				t.Errorf("object value mismatch: %v", err)
			}
		})
	}
}

// Check returns a function that calls t.Fatal if the error is not nil.
func Check(t *testing.T) func(error) {
	return func(err error) {
		if err != nil {
			t.Fatalf("check failed; err != nil: %v", err)
		}
	}
}

// ErrorValue returns the string value of err.
// If err is nil the string "nil" is returned.
func ErrorValue(err error) string {
	if err == nil {
		return "nil"
	}
	return err.Error()
}

// ErrorContains returns true if the string value of subErr is found in the string value of err.
func ErrorContains(err, subErr error) bool {
	return strings.Contains(ErrorValue(err), ErrorValue(subErr))
}
