// Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.

package rpcv0

import (
	"CoralGoCodec/codec"
	"CoralRPCGoSupport/internal/roundtrip"
	cjson "CoralRPCGoSupport/rpc/encoding/json"
	"encoding/json"
	"fmt"

	"github.com/pkg/errors"
)

// rpcInput is the on-the-wire JSON format that is expected.
//
// Input parsing is delayed until we know what struct we will Unmarshal into.
type rpcInput struct {
	Operation string
	Service   string
	Input     json.RawMessage
}

// rpcOutput is the on-the-wire JSON format. It is assumed that some other
// Marshaling process has Marshaled the output as the Output field is just
// raw bytes
type rpcOutput struct {
	Output  json.RawMessage
	Version string
}

// Marshal and Unmarshal satisfy the coral/codec interface

// Marshal defines how to encode an object into the format used on the wire.
func (c RPCv0) Marshal(obj interface{}) ([]byte, error) {
	return cjson.Marshal(obj)
}

// MarshalOutput marshals the coral model and wraps the resulting []byte in
// a rpcOutput
func (c RPCv0) MarshalOutput(obj interface{}) ([]byte, error) {
	out, err := cjson.Marshal(obj)

	if err != nil {
		return nil, errors.Wrap(err, "could not marshal coral obj")
	}

	b, err := json.Marshal(
		&rpcOutput{Output: out, Version: "1.0"},
	)

	if err != nil {
		return nil, errors.Wrap(err, "could not marshal rpcOutput")
	}

	return b, nil
}

// MarshalInput marshals the Coral model and then wraps the resulting []byte
// in rpcInput
func (c RPCv0) MarshalInput(obj interface{}, operation, service string) ([]byte, error) {
	in, err := c.Marshal(obj)

	if err != nil {
		return nil, errors.Wrap(err, "could not marshal coral obj")
	}

	b, err := json.Marshal(
		&rpcInput{Operation: operation, Service: service, Input: in},
	)

	if err != nil {
		return nil, errors.Wrap(err, "could not marshal rpcInput")
	}

	return b, nil
}

// Unmarshal satisfies codec.Server.  It defines how to decode an object from the
// format used on the wire.
func (c RPCv0) Unmarshal(d []byte, obj interface{}) error {
	return cjson.Unmarshal(d, obj, "")
}

// UnmarshalWithService uses the given service in conjunction with the value from the
// "__type" entry in the given map to try to determine a more specific shape than what
// is in the given obj.  This only works when obj is an interface.
func (c RPCv0) UnmarshalWithService(m map[string]interface{}, obj interface{}, service string) error {
	return cjson.UnmarshalMap(m, obj, service)
}

// UnmarshalInput unmarshals an object formatted as rpcInput
func (c RPCv0) UnmarshalInput(data []byte, obj interface{}) error {
	in := &rpcInput{}
	err := json.Unmarshal(data, in)

	if err != nil {
		return errors.Wrap(err, "could not unmarshal data into rpcInput")
	}

	return c.Unmarshal(in.Input, obj)
}

// MarshalRequest satisfies roundtrip.Codec.
func (c RPCv0) MarshalRequest(r *codec.Request) ([]byte, error) {
	opName := fmt.Sprintf("%s#%s", r.Operation.AsmName, r.Operation.ShapeName)
	svcName := fmt.Sprintf("%s#%s", r.Service.AsmName, r.Service.ShapeName)
	return c.MarshalInput(r.Input, opName, svcName)
}

// UnmarshalResponse satisfies roundtrip.Codec.
func (c RPCv0) UnmarshalResponse(rtCtx roundtrip.Context, data []byte, obj interface{}, service string) error {
	out := &rpcOutput{}
	err := json.Unmarshal(data, out)

	if err != nil {
		return roundtrip.NewError("could not unmarshal data into rpcOutput", err, rtCtx)
	}

	// Return the raw error since the error may be from the model.
	return cjson.Unmarshal(out.Output, obj, service)
}
