// Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.

// Package rpcv0 implents a codec that satisfies coral/codec.Codec, coral/codec.RoundTripper,
// and coral/codec.Server
//
// RPCv0 conforms to the on-the-wire protocol specified by the following link:
// https://w.amazon.com/index.php/Coral/Manual/Protocols#RPCv0
package rpcv0

import (
	"CoralGoCodec/codec"
	"CoralRPCGoSupport/internal/roundtrip"
	"CoralRPCGoSupport/rpc"
	"aaa"
	"authv4"
	"authv4/arps"

	"golang.a2z.com/cloudauth"
)

var _ codec.Codec = RPCv0{}
var _ codec.RoundTripper = RPCv0{}
var _ codec.Server = RPCv0{}
var _ roundtrip.Codec = RPCv0{}

// Option is a function that can be provided to configure an RPCv0 codec
type Option func(*RPCv0)

// RPCv0 satisfies the codec.Codec, codec.RoundTripper, and codec.Server interfaces.
//
// Unless otherwise specified these values default to the standard Go default values for their types.
type RPCv0 struct {
	Host               string                        // host address for service. Only Clients need to set this value
	Path               string                        // Defaults to "/"
	SignerV4           *authv4.Signer                // optional
	ARPSAuthorizer     *arps.ARPSAuthorizer          // optional
	AAAClient          aaa.Client                    // optional
	AAA                aaa.Support                   // optional
	RequestIdGenerator rpc.RequestIdGenerator        // optional
	bearerTokenVendor  roundtrip.BearerTokenVendor   // optional
	cloudAuth          cloudauth.AuthSessionProvider // optional
	tripper            roundtrip.Tripper             // created by New()
}

// New returns an initialized RPCv0 struct. This function is appropriate for instantiating codecs for
// server implementations. Creates the infrastructure necessary to perform round trip requests.
func New(options ...Option) RPCv0 {
	v0 := RPCv0{}

	return new(v0, options...)
}

// NewClientCodec returns an RPCv0 initialized with a host. This function is appropriate for instantiating
// codecs for clients. Creates the infrastructure necessary to perform round trip requests.
func NewClientCodec(host string, options ...Option) RPCv0 {
	v0 := RPCv0{
		Host: host,
	}

	return new(v0, options...)
}

// SetSignerV4 is an option function to set RPCv0 Signer
func SetSignerV4(signer *authv4.Signer) Option {
	return (Option)(func(v0 *RPCv0) {
		v0.SignerV4 = signer
	})
}

// SetARPSAuthorizer is an option function to set RPCv0 sigv4 Authorizer
func SetARPSAuthorizer(authorizer *arps.ARPSAuthorizer) Option {
	return (Option)(func(v0 *RPCv0) {
		v0.ARPSAuthorizer = authorizer
	})
}

// SetAAAClient is a proxy for SetAAAForClient.
func SetAAAClient(aaa aaa.Client) Option {
	return SetAAAForClient(aaa)
}

// SetAAAForClient is an option function to set a RPCv0 AAA client.  Use this version
// when you only need client support.  This can be set to either the Roadside Assist or
// the Security Daemon client.
func SetAAAForClient(aaa aaa.Client) Option {
	return (Option)(func(v0 *RPCv0) {
		v0.AAAClient = aaa
	})
}

// SetAAAForServer is an option function to set full AAA support.  Use this version
// when you need Server support.  This can only be set to the Security Daemon client.
func SetAAAForServer(aaa aaa.Support) Option {
	return (Option)(func(v0 *RPCv0) {
		v0.AAA = aaa
	})
}

// SetRequestIdGenerator is an option function that will be used for generating client request ids.
// Amazon standard is to use a UUID v4 identifier for request ids.
func SetRequestIdGenerator(generator rpc.RequestIdGenerator) Option {
	return Option(func(v0 *RPCv0) {
		v0.RequestIdGenerator = generator
	})
}

// SetBearerTokenVendor is an option function to set Bearer tokens in HTTP
// requests. CloudAuth for example requires Bearer token to authenticate a client.
func SetBearerTokenVendor(bearerTokenVendor roundtrip.BearerTokenVendor) Option {
	return (Option)(func(v0 *RPCv0) {
		v0.bearerTokenVendor = bearerTokenVendor
	})
}

// SetCloudAuthForServer is an option function to set full CloudAuth support.
func SetCloudAuthForServer(cloudAuth cloudauth.AuthSessionProvider) Option {
	return (Option)(func(v0 *RPCv0) {
		v0.cloudAuth = cloudAuth
	})
}

// new is the main initializer that the exported initalizers call. This functions ensures
// the default values are set appropriately
func new(v0 RPCv0, options ...Option) RPCv0 {
	for _, option := range options {
		option(&v0)
	}

	if v0.Path == "" {
		v0.Path = "/"
	}

	v0.tripper = roundtrip.Tripper{
		Codec:              v0,
		Host:               v0.Host,
		RequestIdGenerator: v0.RequestIdGenerator,
		AAAClient:          v0.AAAClient,
		ARPSAuthorizer:     v0.ARPSAuthorizer,
		BearerTokenVendor:  v0.bearerTokenVendor,
		SignerV4:           v0.SignerV4,
	}

	return v0
}
