// Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.

package rpcv0

import (
	"CoralGoCodec/codec"
	"CoralGoModel/model"
	cjson "CoralRPCGoSupport/rpc/encoding/json"
	"GoLog/log"
	"aaa"
	"encoding/json"
	"net/http"
	"reflect"
	"strings"

	"github.com/pkg/errors"
	"golang.a2z.com/cloudauth"
)

const (
	headerAccept            = "Accept"
	headerAmznAuthorization = aaa.AaaAuthHeader
	headerAmznDate          = "X-Amz-Date"
	headerAmznTarget        = "X-Amz-Target"
	headerContentType       = "Content-Type"
	headerContentEncoding   = "Content-Encoding"

	contentType     = "application/json"
	contentEncoding = "amz-"
)

// IsSupported satisfies codec.Server
//
// In order to truly check if a request is supported by this codec we need access to the
// request body. This is just a shallow check to satisfy the codec.Server interface.
// UnmarshalRequest will do a more a deeper check and return an appropriate error.
func (c RPCv0) IsSupported(g codec.Getter) bool {
	return isContentTypeSupported(g) && isContentEncodingSupported(g)
}

func isContentTypeSupported(g codec.Getter) bool {
	ct := g.Get(headerContentType)

	if ct == "" {
		return false
	}

	// Content-Type can look like application/json; charset=UTF-8
	ct = strings.Split(ct, ";")[0]

	return ct == contentType
}

func isContentEncodingSupported(g codec.Getter) bool {
	// RPVc1 requires the Content-Encoding header to be amz-1.0 whereas this codec
	// has no strict preference.
	supported := !strings.HasPrefix(g.Get(headerContentEncoding), contentEncoding)
	log.Trace("RPCv0: isContentEncodingSupported", supported)
	return supported
}

// cloudAuthContext is used to track whether or not a particular request requires
// a bearerChallenge before it will be allowed to be retried.
type cloudAuthContext struct {
	bearerChallenge string
}

// UnmarshalRequest reads an http.Request to produce a codec.Request struct with the
// Service, Operation, and Input decoded. It is possible to return a pointer to a codec
// request and an error in the scenario where a CloudAuth authorization failure occurs
// and a Bearer Challenge is necessary. The Codec Request will contain the Bearer Challenge
// which is used when Marshalling the response.
func (c RPCv0) UnmarshalRequest(r *http.Request) (*codec.Request, error) {
	var sctx *aaa.ServiceContext
	var err error

	// Since there is the potential to have two different authorization modes, AAA and CloudAuth,
	// active for a service at the same time we need to be able to distinguish when a request is
	// meant for one vs the other.  We use the presence of the "x-amzn-Authorization" header to
	// signal that a request is using AAA. If the request is using CloudAuth it will instead
	// contain the "Authorization" header.
	hasAAAHeader := r.Header.Get(headerAmznAuthorization) != ""
	if c.AAA != nil && hasAAAHeader {
		if sctx, err = c.AAA.DecodeRequest(r); err != nil {
			return nil, err
		}
		log.Tracef("Decoded ServiceContext %#v", *sctx)
	}

	// TODO: Authentication and authorization for sigv4 must be split and put in interfaces.
	if c.ARPSAuthorizer != nil {
		auth, err := c.ARPSAuthorizer.Authenticate(r)
		if err != nil {
			return nil, errors.Errorf("request is not authorized. error message is %v", err)
		}
		/*
			TODO: This message must be more informative(contain user principal instead of just account Id)
			once Authentication and authorization for sigv4 are split.
		*/
		log.Tracef("Request from account %s is authorized", auth.AccountId())
	}

	// This code between this comment and the next is (mostly) unique
	// to RPCv0
	var rawInput rpcInput
	defer r.Body.Close()
	err = json.NewDecoder(r.Body).Decode(&rawInput)

	if err != nil {
		return nil, errors.Wrap(err, "There was an issue decoding the JSON in the request body")
	}

	if rawInput.Operation == "" || rawInput.Service == "" {
		return nil, errors.New(
			"Invalid request. Expected Operation and Service JSON attributes in the request body",
		)
	}

	opName := getOperationName(rawInput.Operation)
	svcName := getServiceName(rawInput.Service)
	asmName := getAssemblyName(rawInput.Operation)

	if opName == "" || svcName == "" || asmName == "" {
		return nil, errors.New("Request is not supported by the RPCv0 codec.")
	}

	// Make sure that the auth context has the service and operation name from the request.
	if sctx != nil && sctx.Service == "" {
		sctx.Service = svcName
		sctx.Operation = opName
	}

	// code below here is (mostly) a straight copy from RPCv1
	asm := model.LookupService(svcName).Assembly(asmName)
	op, err := asm.Op(opName)
	if err != nil {
		return nil, errors.Wrapf(err, "Unable to retrieve operation %v", opName)
	}

	cr := &codec.Request{
		Service: codec.ShapeRef{
			AsmName:   asmName,
			ShapeName: svcName,
		},
		Operation: codec.ShapeRef{
			AsmName:   asmName,
			ShapeName: opName,
		},
		AuthCtx: sctx,
	}

	// Now that we have all of the information we need to authorize the request, perform
	// authorization before dispatching to the service.
	// - If an error occurs the Codec Request will be returned as nil.
	// - In the case of authorizing CloudAuth requests, we require context from the request
	//   to generate a challenge response. In this scenario we return the codec request
	//   even when an error occurs.
	cr, err = c.authorizeRequest(cr, r, hasAAAHeader)
	if err != nil {
		return cr, err
	}

	if input := op.Input(); input != nil {
		m, mapErr := cjson.ToMap(rawInput.Input)
		if mapErr != nil {
			return nil, mapErr
		}
		targetType := reflect.TypeOf(input.New()).Elem()
		shape := cjson.DetermineShape(reflect.ValueOf(m), targetType, cr.Service.ShapeName)
		cr.Input = shape.New()
		if err := c.UnmarshalWithService(m, cr.Input, cr.Service.ShapeName); err != nil {
			return nil, err
		}
	}

	if output := op.Output(); output != nil {
		cr.Output = output.New()
	}

	return cr, nil
}

// MarshalResponse writes the Output of the given Request to the given ResponseWriter
// adding headers and encryption as necessary to support the CODEC and security.
// If the given Output is nil, then a status 204 response is returned.
func (c RPCv0) MarshalResponse(w http.ResponseWriter, r *codec.Request) {
	if r == nil {
		log.Fatal("Attempt to MarshalResponse without a codec.Request")
		http.Error(w, "Unable to process request", http.StatusInternalServerError)
		return
	}

	headers := w.Header()
	if cloudAuthCtx, ok := r.AuthCtx.(*cloudAuthContext); ok && cloudAuthCtx.bearerChallenge != "" {
		// Returning a 401 with a WWW-Authenticate header represents the Bearer Challenge.
		// This process is outlined here https://w.amazon.com/bin/view/Dev.CDO/UnifiedAuth/CloudAuth/Design/#HFlow
		w.Header().Set("WWW-Authenticate", cloudAuthCtx.bearerChallenge)
		w.WriteHeader(http.StatusUnauthorized)
		return
	}

	log.Tracef("Marshaling response %#v", *r)

	var body []byte
	var err error

	if r.Output != nil {
		body, err = c.MarshalOutput(r.Output)
		if err != nil {
			log.Fatalf("Error marshalling output: %+v", err)
			http.Error(w, "Unable to process response", http.StatusInternalServerError)
			return
		}
		log.Tracef("JSON encoded output %s", body)
	}

	headers.Set(headerContentType, contentType)

	if sctx, ok := r.AuthCtx.(*aaa.ServiceContext); ok && c.AAA != nil {
		body, err = c.AAA.EncodeResponse(sctx, headers, body)
		if err != nil {
			log.Fatalf("Error encoding response using AAA: %+v", err)
			http.Error(w, "Unable to encode response", http.StatusInternalServerError)
			return
		}
		log.Trace("Output has been AAA encoded")
	}

	if body != nil {
		_, err = w.Write(body)

		if err != nil {
			log.Fatalf("Unable to write body of response: %+v", err)
			w.WriteHeader(http.StatusInternalServerError)
		}
	} else {
		w.WriteHeader(http.StatusNoContent)
	}
}

// authorizeRequest performs authorization if the codec was set up to use AAA and/or CloudAuth.
// If neither has been set up then no action is performed.
func (c RPCv0) authorizeRequest(cr *codec.Request, r *http.Request, hasAAAHeader bool) (*codec.Request, error) {
	if c.AAA != nil || c.cloudAuth != nil {
		switch {
		// If AAA was specified and we see the AAA Authorization header, we assume request was made via AAA.
		// Otherwise we treat it as a request that requires CloudAuth
		case c.AAA != nil && hasAAAHeader:
			auth, err := c.AAA.AuthorizeRequest(cr.AuthCtx.(*aaa.ServiceContext))
			if err != nil {
				return nil, err
			}
			if !auth.Authorized {
				return nil, errors.New("Request is not authorized.  Error message is " + auth.ErrorMessage)
			}
			log.Trace("RPCv0: Request is authorized via AAA with code", auth.AuthorizationCode)
		case c.cloudAuth != nil:
			cloudAuthContext := &cloudAuthContext{}
			cr.AuthCtx = cloudAuthContext
			auth, err := c.cloudAuth.AuthorizeRequest(r, cr.Service.ShapeName, cr.Operation.ShapeName)
			if err != nil {
				return nil, err
			}
			if auth.Result == cloudauth.ResultDeny || auth.Result == cloudauth.ResultChallenge {
				cloudAuthContext.bearerChallenge = auth.BearerChallenge
				return cr, errors.New("Request is not authorized")
			}
			log.Trace("RPCv0: Request is authorized via CloudAuth")
		default:
			return nil, errors.New("Request is not authorized.")
		}
	}
	return cr, nil
}

func getOperationName(str string) string {
	return splitAndTakeIndex(str, "#", 1)
}

func getServiceName(str string) string {
	return splitAndTakeIndex(str, "#", 1)
}

func getAssemblyName(str string) string {
	return splitAndTakeIndex(str, "#", 0)
}

// splitAndTakeIndex is a not quite generalised function for splitting strings
// It exists as a helper function to make splitting strings like "com.amazon.coral.demo#WeatherService"
// into assembly and service / operation easier
func splitAndTakeIndex(str string, split string, i int) string {
	arr := strings.SplitN(str, split, 2)

	if len(arr) < 2 {
		return ""
	}

	return arr[i]
}
