// Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.

package rpcv0

import (
	"CoralGoCodec/codec"
	"CoralRPCGoSupport/internal/roundtrip"
	"CoralRPCGoSupport/internal/test/fake"
	"CoralRPCGoSupport/internal/test/fakemodel"
	"aaa"
	"bytes"
	"encoding/json"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/pkg/errors"
)

const (
	testListName  = "best_list_ever"
	testService   = "com.amazon.toplistsservice#TopListsService"
	testOperation = "com.amazon.toplistsservice#FindTopList"
)

// This codec was created to support the CoralRubySuperClient. That being said a best effort should be made to support
// all clients using RPCv0 such as the Coral Diver website.
//
// The following is some example metadata pulled from when CoralRubySuperClient is making an HTTP request.
//
// Verb:
//  "POST"
// URI:
// 	http://localhost:50051/
// Content-Type:
//	"application/json; charset=UTF-8"
// Headers:
// {
//       "x-amzn-requestid"=>["69e9cc89-00e0-4183-bef9-c42df8c729de"],
//       "content-type"=>["application/json; charset=UTF-8"],
//       "x-amz-date"=>["Thu, 05 Jan 2017 20:10:08 GMT"],
//       "x-amzn-client-ttl-seconds"=>["5.0"],
//       "x-amz-target"=>["com.amazon.goodreadstoplistsservice.GoodreadsTopListsService.FindTopList"]
// }
// Http Body:
//   "{\"Operation\":\"com.amazon.goodreadstoplistsservice#FindTopList\",
//     \"Service\":\"com.amazon.goodreadstoplistsservice#GoodreadsTopListsService\",
//     \"Input\":{\"name\":\"test_list\",\"location\":\"world\",\"period\":\"all_time\",\"__type\":\"com.amazon.goodreadstoplistsservice#FindTopListInput\"}}",

func TestIsSupported(t *testing.T) {
	c := New()

	// Note on http.Header:
	// It will normalize the header name using textproto.MIMEHeader. So there's no need to check
	// variations like content-type or content-Type as Content-Type will do.
	//
	// https://golang.org/src/net/http/header.go
	tests := []struct {
		in       codec.Getter
		expected bool
	}{
		{in: http.Header{headerContentType: []string{""}, headerAmznDate: []string{""}}, expected: false},
		{in: http.Header{headerContentType: []string{"application/xml"}, headerAmznDate: []string{"2016"}}, expected: false},
		{in: http.Header{headerContentType: []string{"application/json"}, headerAmznDate: []string{""}}, expected: true},
		{in: http.Header{headerContentType: []string{"application/json; charset=UTF-8"}, headerAmznDate: []string{"2016"}}, expected: true},
		{in: http.Header{headerContentType: []string{"application/json"}, headerAmznDate: []string{"Thu, 05 Jan 2017 20:10:08 GMT"}}, expected: true},
		{in: http.Header{headerContentType: []string{"application/json"}, headerAmznDate: []string{"Thu, 05 Jan 2017 20:10:08 GMT"}, headerAmznTarget: []string{"Coral#Target"}}, expected: true},
		{in: http.Header{headerContentType: []string{"application/json"}, headerAmznDate: []string{"Thu, 05 Jan 2017 20:10:08 GMT"}, headerContentEncoding: []string{contentEncoding}}, expected: false},
	}

	for _, test := range tests {
		actual := c.IsSupported(test.in)

		if actual != test.expected {
			t.Errorf(
				"IsSupported: given: %+v expected: %v actual: %v",
				test.in,
				test.expected,
				actual,
			)
		}
	}
}

func TestUnmarshalRequest(t *testing.T) {
	rpc := New()

	b := buildInputBytes(rpc, t)

	req, err := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	if err != nil {
		t.Fatalf("UnmarshalRequest: unable to create test http request: %v", err)
	}

	cr, err := rpc.UnmarshalRequest(req)

	if err != nil {
		t.Error("expected err from UnmarshalRequest to be nil but was", err)
	}

	if v, ok := cr.Input.(fakemodel.FindTopListInput); ok {
		unmarshaledName := v.Name()
		if *unmarshaledName != testListName {
			t.Errorf("expected FindTopListInput field name to be %s but was %s", testListName, *unmarshaledName)
		}
	} else {
		t.Error("expcted codecRequest Input to be of type FindTopListInput")
	}
}

func TestUnmarshalRequest_ARPS(t *testing.T) {
	t.Parallel()
	rpc := New(SetARPSAuthorizer(&fake.ARPSAuthorizer))
	req, err := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(buildInputBytes(rpc, t)))
	if err != nil {
		t.Fatalf("UnmarshalRequest: unable to create test http request: %v", err)
	}
	fake.Signer.Sign(req)
	if _, err := rpc.UnmarshalRequest(req); err != nil {
		t.Error("Unxpected error", err)
	}
}

func TestUnmarshalRequest_AAA(t *testing.T) {
	fakeAAA := &fake.AAA{AuthResult: &aaa.AuthorizationResult{Authorized: true}}
	rpc := New(SetAAAForServer(fakeAAA))

	b := buildInputBytes(rpc, t)

	req, err := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	req.Header.Set(headerAmznAuthorization, "AAA")
	if err != nil {
		t.Fatalf("UnmarshalRequest: unable to create test http request: %v", err)
	}

	cr, err := rpc.UnmarshalRequest(req)

	if err != nil {
		t.Error("expected err from UnmarshalRequest to be nil but was", err)
	}

	if v, ok := cr.Input.(fakemodel.FindTopListInput); ok {
		unmarshaledName := v.Name()
		if *unmarshaledName != testListName {
			t.Errorf("expected FindTopListInput field name to be %s but was %s", testListName, *unmarshaledName)
		}
	} else {
		t.Error("expcted codecRequest Input to be of type FindTopListInput")
	}
}

func TestUnmarshalRequest_Error(t *testing.T) {
	rpc := New()

	b := buildInputBytes(rpc, t)

	missingCommas := bytes.Replace(b, []byte(","), []byte(" "), 3)
	req, _ := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(missingCommas))
	assertUnmarshalRequestError("invalid json", req, rpc, t)

	noService := bytes.Replace(b, []byte("Service"), []byte("Ecivres"), 3)
	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(noService))
	assertUnmarshalRequestError("missing service", req, rpc, t)

	noAsmName := bytes.Replace(b, []byte("com.amazon.toplistsservice#FindTopList"), []byte("#FindTopList"), 1)
	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(noAsmName))
	assertUnmarshalRequestError("missing assembly name", req, rpc, t)

	noRegisteredOp := bytes.Replace(b, []byte("FindTopList"), []byte("FindBestList"), 1)
	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(noRegisteredOp))
	assertUnmarshalRequestError("operation not registered", req, rpc, t)
}

func TestUnmarshalRequest_Error_AAA(t *testing.T) {
	fakeAAA := &fake.AAA{AuthResult: &aaa.AuthorizationResult{Authorized: true}}
	rpc := New(SetAAAForServer(fakeAAA))

	b := buildInputBytes(rpc, t)

	missingCommas := bytes.Replace(b, []byte(","), []byte(" "), 3)
	req, _ := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(missingCommas))
	req.Header.Set(headerAmznAuthorization, "AAA")
	assertUnmarshalRequestError("invalid json", req, rpc, t)

	noService := bytes.Replace(b, []byte("Service"), []byte("Ecivres"), 3)
	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(noService))
	req.Header.Set(headerAmznAuthorization, "AAA")
	assertUnmarshalRequestError("missing service", req, rpc, t)

	noAsmName := bytes.Replace(b, []byte("com.amazon.toplistsservice#FindTopList"), []byte("#FindTopList"), 1)
	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(noAsmName))
	req.Header.Set(headerAmznAuthorization, "AAA")
	assertUnmarshalRequestError("missing assembly name", req, rpc, t)

	noRegisteredOp := bytes.Replace(b, []byte("FindTopList"), []byte("FindBestList"), 1)
	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(noRegisteredOp))
	req.Header.Set(headerAmznAuthorization, "AAA")
	assertUnmarshalRequestError("operation not registered", req, rpc, t)

	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	req.Header.Set(headerAmznAuthorization, "AAA")
	err := errors.New("Boom")
	fakeAAA.DecodeErr = err
	assertUnmarshalRequestError("error AAA decoding", req, rpc, t)

	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	req.Header.Set(headerAmznAuthorization, "AAA")
	fakeAAA.AuthErr, fakeAAA.DecodeErr = err, nil
	assertUnmarshalRequestError("error AAA authorizing", req, rpc, t)

	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	req.Header.Set(headerAmznAuthorization, "AAA")
	fakeAAA.AuthErr, fakeAAA.AuthResult = nil, &aaa.AuthorizationResult{Authorized: false}
	assertUnmarshalRequestError("not AAA authorized", req, rpc, t)

	req, _ = http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	fakeAAA.AuthErr, fakeAAA.AuthResult = nil, &aaa.AuthorizationResult{Authorized: true}
	assertUnmarshalRequestError("no AAA header present", req, rpc, t)
}

func TestMarshalResponse(t *testing.T) {
	cr := buildOutputRequest()
	rpc := New()
	w := httptest.NewRecorder()
	rpc.MarshalResponse(w, cr)

	if w.Code != http.StatusOK {
		t.Error("expected status code from MarshalResponse to be 200 but was", w.Code)
		t.Log(w.Body.String())
	}

	respOutput := fakemodel.NewFindTopListOutput()
	body, err := ioutil.ReadAll(w.Body)

	if err != nil {
		t.Fatal("expected ReadAll error to be nil but was", err)
	}

	err = rpc.UnmarshalResponse(roundtrip.Context{}, body, &respOutput, "")

	if err != nil {
		t.Fatal("expected UnmarshalOutput error to be nil but was", err)
	}

	respName := respOutput.Name()
	if *respName != testListName {
		t.Errorf("expected response output field Name to be %s but was %v", testListName, respOutput.Name())
		t.Logf("respOutput: %+v", respOutput)
	}

	// No content response since there is not output.
	w = httptest.NewRecorder()
	rpc.MarshalResponse(w, &codec.Request{})
	rpc.MarshalResponse(w, cr)
	if w.Code != http.StatusNoContent {
		t.Error("expected status code from MarshalResponse to be 204 but was", w.Code)
		t.Log(w.Body.String())
	}
}

func TestMarshalResponse_AAA(t *testing.T) {
	cr := buildOutputRequest()
	fakeAAA := &fake.AAA{}
	rpc := New(SetAAAForServer(fakeAAA))
	w := httptest.NewRecorder()
	rpc.MarshalResponse(w, cr)

	if w.Code != http.StatusOK {
		t.Error("expected status code from MarshalResponse to be 200 but was", w.Code)
		t.Log(w.Body.String())
	}

	respOutput := fakemodel.NewFindTopListOutput()
	body, err := ioutil.ReadAll(w.Body)

	if err != nil {
		t.Fatal("expected ReadAll error to be nil but was", err)
	}

	err = rpc.UnmarshalResponse(roundtrip.Context{}, body, &respOutput, "")

	if err != nil {
		t.Fatal("expected UnmarshalOutput error to be nil but was", err)
	}

	respName := respOutput.Name()
	if *respName != testListName {
		t.Errorf("expected response output field Name to be %s but was %v", testListName, respOutput.Name())
		t.Logf("respOutput: %+v", respOutput)
	}

	// No content response since there is not output.
	w = httptest.NewRecorder()
	rpc.MarshalResponse(w, &codec.Request{})
	rpc.MarshalResponse(w, cr)
	if w.Code != http.StatusNoContent {
		t.Error("expected status code from MarshalResponse to be 204 but was", w.Code)
		t.Log(w.Body.String())
	}
}

func TestMarshalResponse_Error(t *testing.T) {
	rpc := New()
	assertMarshalResponseError("nil codec.Request", rpc, nil, t)
	cr := &codec.Request{Output: 0}
	assertMarshalResponseError("invalid output type", rpc, cr, t)
}

func TestMarshalResponse_Error_AAA(t *testing.T) {
	fakeAAA := &fake.AAA{}
	rpc := New(SetAAAForServer(fakeAAA))
	assertMarshalResponseError("nil codec.Request", rpc, nil, t)
	cr := &codec.Request{Output: 0}
	assertMarshalResponseError("invalid output type", rpc, cr, t)
	cr = buildOutputRequest()
	fakeAAA.EncodeErr = errors.New("Boom")
	assertMarshalResponseError("AAA encode error", rpc, cr, t)
}

func TestUnmarshalRequest_CloudAuth(t *testing.T) {
	cloudAuth := &fake.CloudAuth{Operation: "FindTopList", Service: "TopListsService"}
	rpc := New(SetCloudAuthForServer(cloudAuth))

	b := buildInputBytes(rpc, t)

	req, errRequest := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	if errRequest != nil {
		t.Fatalf("UnmarshalRequest: unable to create test http request: %+v", errRequest)
	}

	cr, errUnmarshal := rpc.UnmarshalRequest(req)

	if errUnmarshal != nil {
		t.Fatalf("expected err from UnmarshalRequest to be nil but was: %+v", errUnmarshal)
	}

	v, ok := cr.Input.(fakemodel.FindTopListInput)
	if !ok {
		t.Fatal("expected codecRequest Input to be of type FindTopListInput")
	}
	unmarshaledName := v.Name()
	if *unmarshaledName != testListName {
		t.Fatalf("expected FindTopListInput field name to be %s but was %s", testListName, *unmarshaledName)
	}
}

func TestUnmarshalRequest_CloudAuth_Challenged(t *testing.T) {
	cloudAuth := &fake.CloudAuth{Service: "test", Operation: "test"}
	rpc := New(SetCloudAuthForServer(cloudAuth))

	b := buildInputBytes(rpc, t)

	req, errRequest := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	if errRequest != nil {
		t.Fatalf("UnmarshalRequest: unable to create test http request: %v", errRequest)
	}

	cr, errUnmarshal := rpc.UnmarshalRequest(req)

	if errUnmarshal == nil {
		t.Fatal("Expected error response")
	}
	cloudAuthCtx, ok := cr.AuthCtx.(*cloudAuthContext)
	if !ok {
		t.Fatal("AuthCtx should be of type cloud auth context")
	}
	if cloudAuthCtx.bearerChallenge != "challenge" {
		t.Fatalf("Challenge response was expected to be challenge but was: %v", cloudAuthCtx.bearerChallenge)
	}
}

func TestUnmarshalRequest_CloudAuth_Error(t *testing.T) {
	expectedErr := errors.New("expected error")
	cloudAuth := &fake.CloudAuth{Err: expectedErr}
	rpc := New(SetCloudAuthForServer(cloudAuth))

	b := buildInputBytes(rpc, t)

	req, err := http.NewRequest(http.MethodPost, "/", bytes.NewBuffer(b))
	if err != nil {
		t.Fatalf("UnmarshalRequest: unable to create test http request: %+v", err)
	}

	if _, err = rpc.UnmarshalRequest(req); err != expectedErr {
		t.Fatalf("Expected err %+v but received err %+v", expectedErr, err)
	}
}

func TestMarshalResponse_CloudAuth_Challenge(t *testing.T) {
	cloudAuth := &fake.CloudAuth{}
	rpc := New(SetCloudAuthForServer(cloudAuth))
	cr := &codec.Request{Output: 0}
	ctx := &cloudAuthContext{bearerChallenge: "challenge"}
	cr.AuthCtx = ctx
	w := httptest.NewRecorder()
	rpc.MarshalResponse(w, cr)
	if w.Code != http.StatusUnauthorized {
		t.Error("returns unauthorized with bearer challenge", "- Expected status", http.StatusUnauthorized, "but found", w.Code)
	}
	challengeHeader := w.Header().Get("WWW-Authenticate")
	if challengeHeader != "challenge" {
		t.Error("returns unauthorized with bearer challenge", "- Expected challenge", "challenge", "but found", challengeHeader)
	}
}

func TestMarshalResponse_CloudAuth_WithAAAPresent(t *testing.T) {
	cr := buildOutputRequest()
	fakeAAA := &fake.AAA{}
	cloudAuth := &fake.CloudAuth{}
	rpc := New(SetAAAForServer(fakeAAA), SetCloudAuthForServer(cloudAuth))
	w := httptest.NewRecorder()
	ctx := &cloudAuthContext{}
	cr.AuthCtx = ctx

	rpc.MarshalResponse(w, cr)
	if w.Code != http.StatusOK {
		t.Error("expected status code from MarshalResponse to be 200 but was", w.Code)
		t.Log(w.Body.String())
	}

	respOutput := fakemodel.NewFindTopListOutput()
	body, err := ioutil.ReadAll(w.Body)

	if err != nil {
		t.Fatal("expected ReadAll error to be nil but was", err)
	}

	err = rpc.UnmarshalResponse(roundtrip.Context{}, body, &respOutput, "")

	if err != nil {
		t.Fatal("expected UnmarshalOutput error to be nil but was", err)
	}

	respName := respOutput.Name()
	if *respName != testListName {
		t.Errorf("expected response output field Name to be %s but was %v", testListName, respOutput.Name())
		t.Logf("respOutput: %+v", respOutput)
	}

	// No content response since there is not output.
	w = httptest.NewRecorder()
	rpc.MarshalResponse(w, &codec.Request{})
	rpc.MarshalResponse(w, cr)
	if w.Code != http.StatusNoContent {
		t.Error("expected status code from MarshalResponse to be 204 but was", w.Code)
		t.Log(w.Body.String())
	}
}

func TestGetOperationName(t *testing.T) {
	tests := []struct {
		in       string
		expected string
	}{
		{in: "com.amazon.coral.demo#WeatherReport", expected: "WeatherReport"},
		{in: "com.amazon.coral.demo.WeatherReport", expected: ""},
		{in: "com.amazon.coral.demo#WeatherService#WeatherReport", expected: "WeatherService#WeatherReport"},
	}

	for _, test := range tests {
		actual := getOperationName(test.in)
		if actual != test.expected {
			t.Errorf("getOperationName: given %s expected %s actual %s", test.in, test.expected, actual)
		}
	}
}

func TestGetServiceName(t *testing.T) {
	tests := []struct {
		in       string
		expected string
	}{
		{in: "com.amazon.coral.demo#WeatherService", expected: "WeatherService"},
		{in: "com.amazon.coral.demo.WeatherService", expected: ""},
		{in: "com.amazon.coral.demo#WeatherService#WeatherService", expected: "WeatherService#WeatherService"},
	}

	for _, test := range tests {
		actual := getServiceName(test.in)
		if actual != test.expected {
			t.Errorf("getServiceName: given %s expected %s actual %s", test.in, test.expected, actual)
		}
	}
}

func TestGetAssemblyName(t *testing.T) {
	tests := []struct {
		in       string
		expected string
	}{
		{in: "com.amazon.coral.demo#WeatherReport", expected: "com.amazon.coral.demo"},
		{in: "com.amazon.coral.demo.WeatherReport", expected: ""},
		{in: "com.amazon.coral.demo#WeatherService#WeatherReport", expected: "com.amazon.coral.demo"},
	}

	for _, test := range tests {
		actual := getAssemblyName(test.in)
		if actual != test.expected {
			t.Errorf("getAssemblyName: given %s expected %s actual %s", test.in, test.expected, actual)
		}
	}
}

func buildInputBytes(rpc RPCv0, t *testing.T) []byte {
	topListInput := fakemodel.NewFindTopListInput()
	n := testListName
	topListInput.SetName(&n)
	topListInputBytes, err := rpc.Marshal(topListInput)

	if err != nil {
		t.Fatalf("UnmarshalRequest: could not marshal json for topListInput: %v", err)
	}

	input := rpcInput{
		Service:   testService,
		Operation: testOperation,
		Input:     topListInputBytes,
	}

	b, err := json.Marshal(&input)

	if err != nil {
		t.Fatalf("UnmarshalRequest: could not marshal json for rpcInput: %v", err)
	}

	return b
}

func buildOutputRequest() *codec.Request {
	output := fakemodel.NewFindTopListOutput()
	name := testListName
	output.SetName(&name)
	cr := &codec.Request{
		Output:  output,
		AuthCtx: &aaa.ServiceContext{Service: testService, Operation: testOperation},
	}
	return cr
}

func assertUnmarshalRequestError(name string, req *http.Request, rpc RPCv0, t *testing.T) {
	if cr, err := rpc.UnmarshalRequest(req); cr != nil || err == nil {
		t.Errorf("%s - Expected nil request and an error but found %#v %+v", name, cr, err)
	}
}

func assertMarshalResponseError(name string, rpc RPCv0, cr *codec.Request, t *testing.T) {
	w := httptest.NewRecorder()
	if rpc.MarshalResponse(w, cr); w.Code != http.StatusInternalServerError {
		body := ""
		if w.Body != nil {
			body = w.Body.String()
		}
		t.Error(name, "- Expected status", http.StatusInternalServerError, "but found", w.Code, body)
	}

}
