/* Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package rpcv1

import (
	"CoralGoCodec/codec"
	"CoralRPCGoSupport/internal/roundtrip"
	"CoralRPCGoSupport/rpc"
	"aaa"
	"authv4"
	"authv4/arps"

	"golang.a2z.com/cloudauth"
)

type BasicAuth struct {
	Username, Password string
}

// Assert that we fulfill the interfaces.
var _ codec.Codec = RPCv1{}
var _ codec.RoundTripper = RPCv1{}
var _ codec.Server = RPCv1{}
var _ roundtrip.Codec = RPCv1{}

// https://w.amazon.com/?Coral/Protocols#RPCv1
type RPCv1 struct {
	Host               string                        // required, host or host:port
	Path               string                        // optional
	SignerV4           *authv4.Signer                // optional
	ARPSAuthorizer     *arps.ARPSAuthorizer          // optional
	AuthInfo           *BasicAuth                    // optional
	SecurityToken      string                        // optional, see https://w.amazon.com/index.php/Coral/Specifications/HttpSecurityToken
	AAAClient          aaa.Client                    // optional
	AAA                aaa.Support                   // optional
	RequestIdGenerator rpc.RequestIdGenerator        // optional
	bearerTokenVendor  roundtrip.BearerTokenVendor   // optional
	cloudAuth          cloudauth.AuthSessionProvider // optional
	tripper            roundtrip.Tripper             // created by New()
}

// Represents a construction option.
type Option func(*RPCv1)

// Example: rpc := rpcv1.New("foo.com:8080", rpcv1.SetSignerV4(signer))
// New creates the infrastructure necessary to perform round trip requests.
func New(host string, options ...Option) RPCv1 {
	v1 := RPCv1{Host: host}

	for _, option := range options {
		if option != nil {
			option(&v1)
		}
	}

	v1.tripper = roundtrip.Tripper{
		Codec:              v1,
		Host:               v1.Host,
		RequestIdGenerator: v1.RequestIdGenerator,
		AAAClient:          v1.AAAClient,
		ARPSAuthorizer:     v1.ARPSAuthorizer,
		BearerTokenVendor:  v1.bearerTokenVendor,
		SecurityToken:      v1.SecurityToken,
		SignerV4:           v1.SignerV4,
	}
	if v1.AuthInfo != nil {
		v1.tripper.BasicAuth = &rpc.BasicAuth{Username: v1.AuthInfo.Username, Password: v1.AuthInfo.Password}
	}

	// TODO: It's not possible for this constructor to return an error, but
	//       there is the potential for error cases such as specifying
	//       multiple authentication schemes, e.g. having both SignerV4 and
	//       AAA will cause issues.

	return v1
}

func SetPath(path string) Option {
	return (Option)(func(rpcv1 *RPCv1) {
		rpcv1.Path = path
	})
}

func SetSignerV4(signer *authv4.Signer) Option {
	return (Option)(func(rpcv1 *RPCv1) {
		rpcv1.SignerV4 = signer
	})
}

func SetARPSAuthorizer(authorizer *arps.ARPSAuthorizer) Option {
	return (Option)(func(rpcv1 *RPCv1) {
		rpcv1.ARPSAuthorizer = authorizer
	})
}

func SetBasicAuth(ba *BasicAuth) Option {
	return (Option)(func(rpcv1 *RPCv1) {
		rpcv1.AuthInfo = ba
	})
}

// SetBearerTokenVendor is an option function to set Bearer tokens in HTTP
// requests. CloudAuth for example requires Bearer token to authenticate a client.
func SetBearerTokenVendor(bearerTokenVendor roundtrip.BearerTokenVendor) Option {
	return (Option)(func(v1 *RPCv1) {
		v1.bearerTokenVendor = bearerTokenVendor
	})
}

func SetSecurityToken(securityToken string) Option {
	return (Option)(func(v1 *RPCv1) {
		v1.SecurityToken = securityToken
	})
}

// SetAAAClient is a proxy for SetAAAForClient.
func SetAAAClient(aaa aaa.Client) Option {
	return SetAAAForClient(aaa)
}

// SetAAAForClient is an option function to set a RPCv1 AAA client.  Use this version
// when you only need client support.  This can be set to either the Roadside Assist or
// the Security Daemon client.
func SetAAAForClient(aaa aaa.Client) Option {
	return (Option)(func(v1 *RPCv1) {
		v1.AAAClient = aaa
	})
}

// SetAAAForServer is an option function to set full AAA support.  Use this version
// when you need Server support.  This can only be set to the Security Daemon client.
func SetAAAForServer(aaa aaa.Support) Option {
	return (Option)(func(v1 *RPCv1) {
		v1.AAA = aaa
	})
}

// SetCloudAuthForServer is an option function to set full CloudAuth support.
func SetCloudAuthForServer(cloudAuth cloudauth.AuthSessionProvider) Option {
	return (Option)(func(v1 *RPCv1) {
		v1.cloudAuth = cloudAuth
	})
}

// SetRequestIdGenerator is an option function that will be used for generating client request ids.
// Amazon standard is to use a UUID v4 identifier for request ids.
func SetRequestIdGenerator(generator rpc.RequestIdGenerator) Option {
	return Option(func(v1 *RPCv1) {
		v1.RequestIdGenerator = generator
	})
}
