package amazoncacerts

import (
	"crypto/tls"
	"crypto/x509"
	"errors"
	"io/ioutil"
	"log"
	"net/http"
	"sync"

	"amazon/apollo/envinfo"
)

const (
	internalAndExternalPath = "/etc/internal_and_external_cacerts/"
	internalPath            = "/etc/cacerts/"
)

var (
	ErrMissingCertDirectory     = errors.New("missing Cert Directory")
	internalAndExternalOnce     = sync.Once{}
	internalAndExternalCertPool = &x509.CertPool{}
)

func GetInternalAndExternalCertPool() *x509.CertPool {
	internalAndExternalOnce.Do(func() {
		internalAndExternalCertPool = getInternalAndExternalCertPool()
	})

	return internalAndExternalCertPool
}

var internalOnce sync.Once
var internalCertPool *x509.CertPool

func GetInternalCertPool() *x509.CertPool {
	internalOnce.Do(func() {
		internalCertPool = getInternalCertPool()
	})

	return internalCertPool
}

func getEnvRoot() (envRoot string) {
	env := envinfo.CurEnvironment()
	envRoot, _ = env.Root()

	return envRoot
}

func getInternalAndExternalCertPool() *x509.CertPool {
	internalAndExternalCertDirectory := getEnvRoot() + internalAndExternalPath

	pool, err := loadCertsFromDirectory(internalAndExternalCertDirectory)

	if err != nil {
		if err == ErrMissingCertDirectory {
			rootCAs := x509.NewCertPool()
			rootCAs.AppendCertsFromPEM([]byte(internalExternalPemEncodedAmazonCACerts))
			return rootCAs
		}
		log.Println(err)
		return nil
	}

	return pool
}

func getInternalCertPool() *x509.CertPool {
	internalCertDirectory := getEnvRoot() + internalPath

	pool, err := loadCertsFromDirectory(internalCertDirectory)

	if err != nil {
		if err == ErrMissingCertDirectory {
			rootCAs := x509.NewCertPool()
			rootCAs.AppendCertsFromPEM([]byte(internalPemEncodedAmazonCACerts))
			return rootCAs
		}
		log.Println(err)
		return nil
	}

	return pool
}

func GetHttpClient(internalOnly bool) *http.Client {
	var roots *x509.CertPool

	if internalOnly == true {
		roots = getInternalCertPool()
	} else {
		roots = GetInternalAndExternalCertPool()
	}

	tlsConf := tls.Config{RootCAs: roots}

	tr := &http.Transport{TLSClientConfig: &tlsConf}
	return &http.Client{Transport: tr}

}

func loadCertsFromDirectory(directory string) (roots *x509.CertPool, err error) {
	roots = x509.NewCertPool()

	fis, err := ioutil.ReadDir(directory)

	if err != nil {
		// Tell caller the cert directory is missing so they can handle
		// handle it gracefully
		return nil, ErrMissingCertDirectory
	}

	for _, fi := range fis {
		data, err := ioutil.ReadFile(directory + "/" + fi.Name())

		if err != nil {
			return nil, err
		}

		roots.AppendCertsFromPEM(data)
	}

	return roots, nil
}
