/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package cloudauth

import (
	"GoLog/log"
	"amazoncacerts"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"
	"time"

	"golang.org/x/oauth2"
	"golang.org/x/oauth2/clientcredentials"
)

const (
	// DefaultIssuerURL is CloudAuth's default issuer URL.
	DefaultIssuerURL = "https://oauth.cloudauth.a2z.com"
)

const (
	// defaultRealm is the realm for Auth Server resource.
	defaultRealm = "CloudAuthServer"
	// defaultScopes is the default scopes for Auth Server resource.
	defaultScopes = "https://aaa.amazon.com/scopes/CloudAuthServer#getToken https://aaa.amazon.com/scopes/CloudAuthServer#introspect"
	// 1 MiB is a reasonable limit for response length from Auth Server.
	maxResponseLen           = 1 << 20
	metadataURLSuffix        = "/.well-known/openid-configuration"
	defaultCacheCleaningRate = 20 * time.Minute
)

type authDiscoveryMetadata struct {
	Issuer                string `json:"issuer"`
	AuthorizationEndpoint string `json:"authorization_endpoint"`
	TokenEndpoint         string `json:"token_endpoint"`
	IntrospectionEndpoint string `json:"introspection_endpoint"`
	BootstrapEndpoint     string `json:"bootstrap_endpoint"`
	JwksURI               string `json:"jwks_uri"`
}

// IntrospectResponse is the introspection response from an Auth Server on an access token.
//
// References:
// https://w.amazon.com/bin/view/C2S/CloudAuth/AuthServerApi/#Introspect
// https://tools.ietf.org/html/rfc7662
type IntrospectResponse struct {
	Active         bool     `json:"active"`
	Audience       []string `json:"aud,omitempty"`
	Subject        string   `json:"sub,omitempty"`
	Scope          string   `json:"scope,omitempty"`
	Issuer         string   `json:"iss,omitempty"`
	ClientID       string   `json:"client_id,omitempty"`
	ExpiryUnixTime int64    `json:"exp,omitempty"`
}

// AuthServerTokenProvider is an interface for initiating communication with a
// specific Auth Server. The returned token is to be used with subsequent requests
// to the Auth Server to either request claims against a specific Resource Server
// (see NewClient) or introspect CloudAuth protected requests (see AuthSessionProvider).
//
// Reference: https://w.amazon.com/bin/view/C2S/CloudAuth/ClientSpec/#Authenticating_to_the_Auth_Server
//
type AuthServerTokenProvider interface {
	// Token returns an access token from Auth server obtained
	// through initial client authentication.
	Token(url string) (*oauth2.Token, error)
}

// AuthSessionProvider is an interface that represents a valid session
// with Auth Server.
type AuthSessionProvider interface {
	// WithClient returns a context with key oauth2.HTTPClient set with a
	// http.Client that can be used to send authenticated HTTP requests with
	// Auth Server.
	WithClient(ctx context.Context) context.Context
	// TokenEndpoint returns the URL for the endpoint that are used for
	// token requests for Auth Server.
	TokenEndpoint() string
	// Introspect returns the response from querying Auth Server
	// on the access token received from a CloudAuth-enabled client.
	Introspect(accessToken string) (*IntrospectResponse, error)
	// AuthorizeRequest authorizes a request to a resource server
	// and returns an authorization result.
	// If request is not authorized, result will contain a bearer challenge.
	AuthorizeRequest(r *http.Request, svcName string, opName string) (*AuthorizationResult, error)
}

type AuthorizationResult struct {
	Result          Result
	BearerChallenge string
}

type Result int

const (
	ResultAllow Result = iota
	ResultDeny
	ResultChallenge
)

// authSessionConfig contains various options to configure the behavior
// of the session with Auth Server such as HTTP client to use and the Auth Server URL.
type authSessionConfig struct {
	httpClient        *http.Client
	issuerURL         string
	cacheCleaningRate time.Duration
}

// AuthSessionOption represents a functional option.
type AuthSessionOption func(*authSessionConfig) error

// withAuthServerURL sets the Auth Server URL. Used for internal testing.
func withAuthServerURL(url string) AuthSessionOption {
	return func(config *authSessionConfig) error {
		config.issuerURL = url
		return nil
	}
}

// withAuthSessionClient sets the HTTP client to use. Used for internal testing.
func withAuthSessionClient(client *http.Client) AuthSessionOption {
	return func(config *authSessionConfig) error {
		switch {
		case client == nil:
			return &InvalidError{Type: "HTTP client"}
		case client.Timeout == 0:
			return &InvalidError{Type: "HTTP client", Value: "Timeout is 0"}
		}

		config.httpClient = client
		return nil
	}
}

// WithCleaningRate sets the interval at which to clean expired entries from the cache.
func WithCleaningRate(cleaningRate time.Duration) AuthSessionOption {
	return func(config *authSessionConfig) error {
		config.cacheCleaningRate = cleaningRate
		return nil
	}
}

// authSession is an implementation of AuthSessionProvider.
type authSession struct {
	metadata authDiscoveryMetadata
	client   *http.Client
	cache    introspectionCache
}

type introspectionCache interface {
	get(token string) *IntrospectResponse
	put(token string, resp *IntrospectResponse)
}

func (a *authSession) WithClient(ctx context.Context) context.Context {
	return context.WithValue(ctx, oauth2.HTTPClient, a.client)
}

func (a *authSession) TokenEndpoint() string {
	return a.metadata.TokenEndpoint
}

func (a *authSession) Introspect(accessToken string) (*IntrospectResponse, error) {
	if accessToken == "" {
		return nil, &InvalidError{Type: "access token"}
	}

	if resp := a.cache.get(accessToken); resp != nil {
		return resp, nil
	}

	r, err := a.client.PostForm(a.metadata.IntrospectionEndpoint, url.Values{"token": {accessToken}})
	if err != nil {
		return nil, &GeneralError{err}
	}

	defer r.Body.Close()
	body, err := ioutil.ReadAll(io.LimitReader(r.Body, maxResponseLen))
	if err != nil {
		return nil, &GeneralError{err}
	}

	if r.StatusCode != http.StatusOK {
		return nil, &GeneralError{fmt.Errorf("%d status returned from introspect", r.StatusCode)}
	}

	var resp IntrospectResponse
	if err = json.Unmarshal(body, &resp); err != nil {
		return nil, &GeneralError{err}
	}

	if !resp.Active || time.Now().Unix() > resp.ExpiryUnixTime {
		return nil, &InvalidError{Type: "access token", Value: fmt.Sprintf("got an introspection response from the auth server that is not active or has expired. Active flag is %t expiry unix time is %d", resp.Active, resp.ExpiryUnixTime)}
	}

	a.cache.put(accessToken, &resp)
	return &resp, nil
}

// AuthorizeRequest satisfies the interface described in AuthSessionProvider.
func (a *authSession) AuthorizeRequest(r *http.Request, svcName string, opName string) (*AuthorizationResult, error) {
	if err := isSecureRequest(r); err != nil {
		return nil, err
	}
	requiredScope := generateAAAResourceScope(svcName, opName)
	result := &AuthorizationResult{Result: ResultDeny}
	result.BearerChallenge = createBearerChallenge(a.metadata.IntrospectionEndpoint, svcName, requiredScope)
	authorizationHeader := r.Header.Get("Authorization")
	if authorizationHeader == "" {
		result.Result = ResultChallenge
		return result, nil
	}

	authorizationHeaderParts := strings.Split(authorizationHeader, " ")
	if len(authorizationHeaderParts) != 2 {
		log.Tracef("Unrecognized authorization header %q", authorizationHeader)
		result.Result = ResultChallenge
		return result, nil
	}
	resp, err := a.Introspect(authorizationHeaderParts[1])
	if err != nil {
		switch err.(type) {
		// InvalidError represents something was wrong with the presented Token.
		// E.g. The supplied token was invalid or expired. In this case we want to generate a bearer challenge.
		// Otherwise it can be assumed that an internal server error occurred during introspection
		case *InvalidError:
			result.Result = ResultChallenge
			return result, nil
		default:
			return nil, err
		}
	}
	if strings.Contains(resp.Scope, requiredScope) {
		result.Result = ResultAllow
		result.BearerChallenge = ""
	}

	return result, nil
}

// NewAuthSession establishes a session to Auth Server, it returns an AuthSessionProvider
// interface that can be used to create CloudAuth clients and introspect CloudAuth access
// tokens. tokenProvider needs to be set with a valid AuthServerTokenProvider (currently
// only SigV4-based AuthServerTokenProvider is supported)
//
// NOTE: By default, the call will use a HTTP client loaded with Amazon internal CA certs with a
//       60 second timeout and the default Auth Server URL.
func NewAuthSession(tokenProvider AuthServerTokenProvider, options ...AuthSessionOption) (AuthSessionProvider, error) {
	// Default values
	httpClient := amazoncacerts.GetHttpClient(false)
	httpClient.Timeout = 60 * time.Second

	config := authSessionConfig{httpClient: httpClient, issuerURL: DefaultIssuerURL, cacheCleaningRate: defaultCacheCleaningRate}

	// Apply options
	for _, option := range options {
		if option != nil {
			if err := option(&config); err != nil {
				return nil, err
			}
		}
	}

	var metadata authDiscoveryMetadata

	// Parse the resource URL and extract the FQDN.
	u, err := url.Parse(config.issuerURL)
	if err != nil {
		return nil, &InvalidError{Type: "issuerURL", Value: config.issuerURL}
	}

	uriAuthority := u.Hostname()
	if u.Port() != "" {
		uriAuthority += ":" + u.Port()
	}

	if tokenProvider == nil {
		return nil, &InvalidError{Type: "tokenProvider"}
	}

	r, err := config.httpClient.Get(config.issuerURL + metadataURLSuffix)
	if err != nil {
		return nil, &GeneralError{err}
	}

	if r.StatusCode != http.StatusOK {
		return nil, &GeneralError{fmt.Errorf("Un-expected response code during discovery: %d", r.StatusCode)}
	}

	body, err := ioutil.ReadAll(io.LimitReader(r.Body, maxResponseLen))
	r.Body.Close()
	if err != nil {
		return nil, &GeneralError{err}
	}

	// Parse Authorization Server meta-data.
	if err = json.Unmarshal(body, &metadata); err != nil {
		return nil, &GeneralError{err}
	}

	// Step 1: Perform initial client authentication.
	token, err := tokenProvider.Token(metadata.AuthorizationEndpoint)
	if err != nil {
		return nil, &GeneralError{err}
	}

	//
	// Step 2: Using the access token from initial client authentication,
	//         get an access token for Auth Server using the
	//         two-legged OAuth2.0 client credentials flow.
	//
	credentialsConfig := &clientcredentials.Config{
		TokenURL:  metadata.TokenEndpoint,
		AuthStyle: oauth2.AuthStyleInParams,
		EndpointParams: url.Values{
			"grant_type": {token.TokenType},
			"assertion":  {token.AccessToken},
			"host":       {uriAuthority}, // URI Authority of the issuer
			"scope":      {defaultScopes},
			"realm":      {defaultRealm},
		},
	}

	ctx := context.WithValue(context.Background(), oauth2.HTTPClient, config.httpClient)

	// Return a valid authSession.
	return &authSession{
		metadata: metadata,
		client:   credentialsConfig.Client(ctx),
		cache:    newIntrospectionCache(config.cacheCleaningRate),
	}, nil
}

// createBearerChallenge generates a Bearer Challenge to be used in WWW-Authenticate Header for client to generate auth token.
// For further details check the RFC https://tools.ietf.org/html/rfc6750#section-3.
func createBearerChallenge(introspectionEndpoint string, svcName string, requiredScope string) string {
	return fmt.Sprintf("Bearer iss=\"%s\", realm=%s, scope=\"%s\"", introspectionEndpoint, svcName, requiredScope)
}

func generateAAAResourceScope(serviceName string, operationName string) string {
	return fmt.Sprintf("https://aaa.amazon.com/scopes/%s#%s", serviceName, operationName)
}

// isSecureRequest ensures requests must be made over TLS.
// Check if the request was from a secure socket.
// If it wasn't check if it was proxied request from
// localhost using the forwarded header to ensure it was secure.
// Proxied requests are typically from CDORelay.
func isSecureRequest(r *http.Request) error {
	if r.TLS != nil {
		return nil
	}

	// Check loopback range 127.0.0.1 - 127.255.255.255.
	if !strings.HasPrefix(r.RemoteAddr, "127.") {
		return fmt.Errorf("insecure request made with remote address %q which was not from loopback address", r.RemoteAddr)
	}

	// Check all X-Forwarded-Proto headers to see if any hop was made via an unencrypted channel.
	if httpXForwardedProtoFound(r) {
		return fmt.Errorf("request was mode over insecure channel: Found X-Forwarded-Proto with http as value")
	}

	forwardedElements := parseForwardedHeader(r)

	// Check all Forwarded elements to see if any hop was made via an unencrypted channel.
	for _, element := range forwardedElements {
		if strings.EqualFold(element.proto, "http") {
			return fmt.Errorf("request was mode over insecure channel: Found Forwarded Header with http as proto value")
		}
	}

	// Find last forwarded Header element in a potential list of forwarded headers and use it's proto property to determine if request was proxied securely.
	if len(forwardedElements) == 0 {
		return fmt.Errorf("request was made over insecure channel: No Forwarded Header found")
	}

	lastForwardedElement := forwardedElements[len(forwardedElements)-1]
	if !strings.EqualFold(lastForwardedElement.proto, "https") {
		return fmt.Errorf("request was made over insecure channel: Last Forwarded Header Element proto was %q instead of https", lastForwardedElement.proto)
	}

	return nil
}

type forwardedElement struct {
	proto string
}

// parseForwardedHeader parses all Forwarded elements from a HTTP request into an ordered array of forwarded elements.
// Forwarded Header could look like Forwarded: for=192.0.2.43, for=198.51.100.17;proto=https.
// Check https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Forwarded for further examples.
func parseForwardedHeader(r *http.Request) []forwardedElement {
	forwardedElements := make([]forwardedElement, 0)
	forwardedHeaders := r.Header["Forwarded"]
	for _, header := range forwardedHeaders {
		headerElements := strings.Split(header, ",")
		for _, element := range headerElements {
			elementProperties := strings.Split(element, ";")
			forwardedElement := forwardedElement{}
			for _, property := range elementProperties {
				// TODO: Parse other by, for, and host properties and add them to the list of forwardElements returned.
				// We don't bother with this for now as the other properties just are not needed
				if strings.HasPrefix(property, "proto=") {
					forwardedElement.proto = property[len("proto="):]
				}
			}
			forwardedElements = append(forwardedElements, forwardedElement)
		}
	}
	return forwardedElements
}

// httpXForwardedProtoFound checks for a X-Forwarded-Proto header value that was unencrypted (http).
// It returns true if one was found, false otherwise.
func httpXForwardedProtoFound(r *http.Request) bool {
	xForwardedProtoHeaders := r.Header["X-Forwarded-Proto"]
	for _, header := range xForwardedProtoHeaders {
		headerElements := strings.Split(header, ",")
		for _, element := range headerElements {
			element = strings.TrimSpace(element)
			if strings.EqualFold(element, "http") {
				return true
			}
		}
	}
	return false
}
