/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package cloudauth

import (
	"sync"
	"time"
)

// cache satisfies introspectionCache.
type cache struct {
	mp               map[string]*IntrospectResponse
	mutex            sync.RWMutex
	lastCleaningTime time.Time
	cleaningRate     time.Duration
}

// newIntrospectionCache creates a cache that cleans out expired entries at the given cleaning rate.
func newIntrospectionCache(cleaningRate time.Duration) *cache {
	c := cache{mp: make(map[string]*IntrospectResponse)}
	c.lastCleaningTime = time.Now()
	c.cleaningRate = cleaningRate
	return &c
}

// get returns the introspection result for the given token.
// If the cached token has expired then nil is returned.
func (c *cache) get(token string) *IntrospectResponse {
	c.mutex.RLock()
	entry := c.mp[token]
	c.mutex.RUnlock()

	// It's expected that returning nil for an expired result will cause a fresh value to be retrieved and injected
	// so we don't bother to delete the item.
	if entry == nil || time.Now().Unix() > entry.ExpiryUnixTime {
		return nil
	}
	return entry
}

// put adds an entry to the cache. If it has been longer than the cleaning rate since the last time the cache was cleaned
// then a cleaning of expired entries is triggered.
func (c *cache) put(token string, resp *IntrospectResponse) {
	c.mutex.Lock()
	c.mp[token] = resp

	if time.Now().After(c.lastCleaningTime.Add(c.cleaningRate)) {
		c.lastCleaningTime = time.Now()
		go c.cleanExpired()
	}
	c.mutex.Unlock()
}

// cleanExpired sweeps the cache removing expired entries.
func (c *cache) cleanExpired() {
	c.mutex.Lock()
	currentUnixTime := time.Now().Unix()
	for k, v := range c.mp {
		if currentUnixTime > v.ExpiryUnixTime {
			delete(c.mp, k)
		}
	}
	c.mutex.Unlock()
}
