/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package cloudauth

import (
	"fmt"
	"math"
	"sync"
	"testing"
	"time"
)

func TestCache(t *testing.T) {
	// One nanosecond is used as the cache cleaning rate to ensure all of the expiredTokens are removed from the cache.
	// Using one nanosecond results in a cleaning be triggered whenever an item is added. For the expiredTokens this
	// causes them to be removed.
	cache := newIntrospectionCache(1 * time.Nanosecond)
	activeTokenCount := 10

	// First let's load the cache with a mix of tokens we expect to stay in the cache (infiniteTokens) and tokens we expect to be removed (expiredTokens).
	wg := &sync.WaitGroup{}
	for c := 0; c < activeTokenCount; c++ {
		wg.Add(2)
		go func(tokenNumber int, wg *sync.WaitGroup) {
			cache.put(fmt.Sprintf("expiredToken %d", tokenNumber), &IntrospectResponse{Active: true, ExpiryUnixTime: 0})
			wg.Done()
		}(c, wg)
		go func(tokenNumber int, wg *sync.WaitGroup) {
			cache.put(fmt.Sprintf("infiniteToken %d", tokenNumber), &IntrospectResponse{Active: true, ExpiryUnixTime: math.MaxInt64})
			wg.Done()
		}(c, wg)
	}
	wg.Wait()

	// Then let's kick off requests to get the entries we've added to the cache.
	infiniteTokenRespChan := make(chan *IntrospectResponse, activeTokenCount)
	expiredTokenRespChan := make(chan *IntrospectResponse, activeTokenCount)
	for c := 0; c < activeTokenCount; c++ {
		go func(tokenNumber int, respChan chan *IntrospectResponse) {
			respChan <- cache.get(fmt.Sprintf("infiniteToken %d", tokenNumber))
		}(c, infiniteTokenRespChan)
		go func(tokenNumber int, respChan chan *IntrospectResponse) {
			respChan <- cache.get(fmt.Sprintf("expiredToken %d", tokenNumber))
		}(c, expiredTokenRespChan)
	}

	// Now let's check if the infiniteTokens stayed in the cache.
	var infiniteTokensRemovedFromCache int
	for i := 0; i < activeTokenCount; i++ {
		if resp := <-infiniteTokenRespChan; resp == nil {
			infiniteTokensRemovedFromCache++
		}
	}
	if infiniteTokensRemovedFromCache != 0 {
		t.Fatalf("Expected the infinateTokens to stay in the cache, but found %d that were removed.", infiniteTokensRemovedFromCache)
	}

	// Finally let's check if the expiredTokens got removed.
	var expiredTokensInCache int
	for i := 0; i < activeTokenCount; i++ {
		if resp := <-expiredTokenRespChan; resp != nil {
			expiredTokensInCache++
		}
	}
	if expiredTokensInCache != 0 {
		t.Fatalf("Expected the expiredTokens to be removed from the cache, but found %d that were not.", expiredTokensInCache)
	}
}
