/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package cloudauth

import (
	"context"
	"net/url"

	"golang.org/x/oauth2"
	"golang.org/x/oauth2/clientcredentials"
)

// ClientTokenProvider is an interface that provides an access token
// for a Resource Server.
type ClientTokenProvider interface {
	// Token returns the OAuth2 token for a Resource Server.
	Token() *oauth2.Token
}

// clientToken is an implementation of ClientTokenProvider.
type clientToken struct {
	authSession AuthSessionProvider
	token       *oauth2.Token
}

func (c *clientToken) Token() *oauth2.Token {
	return c.token
}

// ResourceParameter represents the parameters needed to get access token for
// a Resource Server.
type ResourceParameter struct {
	URL   string
	Scope string
	Realm string
}

// ResourceTokenMap is a mapping of FQDN to OAuth2 tokens.
type ResourceTokenMap struct {
	mapping map[string]*oauth2.Token
}

// ResourceTokens is a runtime cache of FQDN to OAuth2 tokens.
var ResourceTokens = &ResourceTokenMap{mapping: make(map[string]*oauth2.Token)}

// Vend provides an access token given a FQDN.
func (r ResourceTokenMap) Vend(host string) (string, error) {
	if token, ok := r.mapping[host]; ok {
		return token.AccessToken, nil
	}

	return "", &NoTokenError{host}
}

// clear removes all the cached access tokens. Only used for testing.
func (r *ResourceTokenMap) clear() {
	for k := range r.mapping {
		delete(r.mapping, k)
	}
}

// NewClient returns a ClientTokenProvider interface that returns an access token
// to access a Resource Server.
func NewClient(param ResourceParameter, authSession AuthSessionProvider) (ClientTokenProvider, error) {
	if authSession == nil {
		return nil, &InvalidError{Type: "Auth Server session"}
	}

	if param.URL == "" {
		return nil, &InvalidError{Type: "Resource URL"}
	}

	if param.Scope == "" {
		return nil, &InvalidError{Type: "Scope"}
	}

	if param.Realm == "" {
		return nil, &InvalidError{Type: "Realm"}
	}

	// Parse the resource URL and extract the FQDN.
	u, err := url.Parse(param.URL)
	if err != nil {
		return nil, &InvalidError{Type: "Resource URL", Value: param.URL}
	}
	host := u.Hostname()

	// Set up a two-legged OAuth2.0 client credentials flow to get
	// an access token for the Resource Server.
	config := &clientcredentials.Config{
		TokenURL:  authSession.TokenEndpoint(),
		AuthStyle: oauth2.AuthStyleInParams,
		EndpointParams: url.Values{
			"host":  {host},
			"scope": {param.Scope},
			"realm": {param.Realm},
		},
	}

	// Use the context from AuthSession that provides an authenticated HTTP client.
	token, err := config.Token(authSession.WithClient(context.Background()))
	if err != nil {
		return nil, &OAuthError{err}
	}

	// Add the FQDN:port into mapping.
	uriAuthority := host
	if u.Port() != "" {
		uriAuthority += ":" + u.Port()
	}

	ResourceTokens.mapping[uriAuthority] = token

	return &clientToken{
		authSession: authSession,
		token:       token,
	}, nil
}
