/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package cloudauth

import (
	"context"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"
)

// MockAuthSession is a mocked implementation of AuthSessionProvider.
type MockAuthSession struct{}

func (a *MockAuthSession) AuthorizeRequest(r *http.Request, svcName string, opName string) (*AuthorizationResult, error) {
	panic("implement me")
}

// WithClient returns a mocked context.
func (a *MockAuthSession) WithClient(ctx context.Context) context.Context {
	return context.Background()
}

// TokenEndpoint returns a mocked token endpoint.
func (a *MockAuthSession) TokenEndpoint() string {
	return "https://foobar.amazon.com/token"
}

// Introspect returns a mocked introspection response.
func (a MockAuthSession) Introspect(accessToken string) (*IntrospectResponse, error) {
	return &IntrospectResponse{Active: false}, nil
}

// TestNewClient_InvalidParams tests on different invalid parameters.
func TestNewClient_InvalidParams(t *testing.T) {
	for _, tc := range []struct {
		name  string
		param ResourceParameter
	}{
		{
			name: "Empty URL",
			param: ResourceParameter{
				Realm: mockServiceName,
				Scope: mockServiceScope,
			},
		},
		{
			name: "Empty Realm",
			param: ResourceParameter{
				URL:   mockServiceURL,
				Scope: mockServiceScope,
			},
		},
		{
			name: "Empty Scope",
			param: ResourceParameter{
				URL:   mockServiceURL,
				Realm: mockServiceName,
			},
		},
		{
			name: "Invalid Resource URL",
			param: ResourceParameter{
				URL:   "http://[::1]:namedport",
				Scope: mockServiceScope,
				Realm: mockServiceName,
			},
		},
	} {
		t.Run(tc.name, func(t *testing.T) {
			_, err := NewClient(tc.param, &MockAuthSession{})

			switch {
			case err == nil:
				t.Error("did not get expected error")
			case err != nil && err.Error() == "":
				t.Error("empty error message")
			case err != nil:
				if _, ok := err.(*InvalidError); !ok {
					t.Errorf("un-expected error type: %T", err)
				}
			}
		})
	}
}

// TestNewClient_NilAuthSession tests when the session with Auth Server is nil.
func TestNewClient_NilAuthSession(t *testing.T) {
	_, err := NewClient(
		ResourceParameter{
			URL:   mockServiceURL,
			Scope: mockServiceScope,
			Realm: mockServiceName,
		}, nil)

	switch {
	case err == nil:
		t.Error("did not get expected error")
	case err != nil && err.Error() == "":
		t.Error("empty error message")
	case err != nil:
		if _, ok := err.(*InvalidError); !ok {
			t.Errorf("un-expected error type: %T", err)
		}
	}
}

// TestNewClient is a test suite of obtaining access tokens for Resource Servers.
func TestNewClient(t *testing.T) {
	tests := []struct {
		description     string
		authServerToken AuthServerTokenProvider
		expectError     bool
		behaviour       serverBehaviour
	}{
		{
			description:     "Token Request (Resource) - Bogus Response",
			authServerToken: &mockAuthServerToken{},
			expectError:     true,
			behaviour:       serverBehaviour{state: stateAuthenticated, response: responseBogus},
		},
		{
			description:     "Token Request (Resource) - 404 Response",
			authServerToken: &mockAuthServerToken{},
			expectError:     true,
			behaviour:       serverBehaviour{state: stateAuthenticated, response: responseInvalidStatusCode},
		},
		{
			description:     "Token Request (Resource) - I/O",
			authServerToken: &mockAuthServerToken{},
			expectError:     true,
			behaviour:       serverBehaviour{state: stateAuthenticated, response: responseIoError},
		},
		{
			description:     "Token Request (Resource) - Valid Response",
			authServerToken: &mockAuthServerToken{},
		},
	}

	for _, c := range tests {
		t.Run(c.description, func(t *testing.T) {
			s := mockCloudAuthServer{t: t, behaviour: c.behaviour}
			ts := httptest.NewTLSServer(s.Handler())
			defer ts.Close()

			ts.Client().Timeout = 1 * time.Minute

			// Flush all the cached access tokens.
			ResourceTokens.clear()

			// Setup a valid session with Auth Server.
			as, err := NewAuthSession(c.authServerToken, withAuthServerURL(ts.URL), withAuthSessionClient(ts.Client()))

			if err != nil {
				t.Fatalf("got unexpected error %+v", err)
			}

			cs, err := NewClient(
				ResourceParameter{
					URL:   mockServiceURL,
					Scope: mockServiceScope,
					Realm: mockServiceName,
				}, as)

			switch {
			case c.expectError && err == nil:
				t.Fatal("did not get expected error")
			case c.expectError && err != nil:
				if err.Error() == "" {
					t.Errorf("empty error message")
				} else if _, ok := err.(*OAuthError); !ok {
					t.Errorf("did not get expected error type: %T", err)
				}

				// Should fail to vend an access token.
				token, err := ResourceTokens.Vend(mockServiceAuthority)

				switch {
				case err == nil:
					t.Errorf("did not get expected error")
				case err.Error() == "":
					t.Errorf("empty error message")
				case token != "":
					t.Errorf("token is not empty")
				default:
					if _, ok := err.(*NoTokenError); !ok {
						t.Errorf("did not get expected error type: %T", err)
					}
				}

				return
			case err != nil:
				t.Errorf("got unexpected error %+v", err)
				return
			}

			// Happy path.
			if want := "resource_server_token"; cs.Token().AccessToken != want {
				t.Errorf("Got token %s, want %s", cs.Token().AccessToken, want)
				return
			}

			// Should be able to vend an expected acccess token.
			token, err := ResourceTokens.Vend(mockServiceAuthority)
			if err != nil {
				t.Errorf("ResourceTokens.Vend failed: %+v", err)
				return
			}

			if want := "resource_server_token"; token != want {
				t.Errorf("Got token %s, want %s", token, want)
			}
		})
	}
}
