/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package cloudauth

import (
	"amazoncacerts"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/aws/credentials"
	sigv4 "github.com/aws/aws-sdk-go/aws/signer/v4"
	"golang.org/x/oauth2"
)

// SigV4Authenticator is an implementation of AuthServerTokenProvider interface
// to authenticate to CloudAuth's Auth Server using SigV4.
type SigV4Authenticator struct {
	// httpClient is the HTTP client used to talk to Auth Server.
	httpClient *http.Client
	// region is the current AWS region.
	region string
	// credentials is the current AWS credentials for the SigV4 request.
	credentials *credentials.Credentials
}

// SigV4Option represents a functional option
type SigV4Option func(*SigV4Authenticator) error

// withSigV4Client sets the HTTP client to use. Used for internal testing
func withSigV4Client(client *http.Client) SigV4Option {
	return func(s *SigV4Authenticator) error {
		switch {
		case client == nil:
			return &InvalidError{Type: "HTTP client"}
		case client.Timeout == 0:
			return &InvalidError{Type: "HTTP client", Value: "Timeout is 0"}
		}

		s.httpClient = client
		return nil
	}
}

// WithSigV4Credentials sets the AWS credentials to use
func WithSigV4Credentials(credentials *credentials.Credentials) SigV4Option {
	return func(s *SigV4Authenticator) error {
		if credentials == nil {
			return &InvalidError{Type: "credentials"}
		}

		s.credentials = credentials
		return nil
	}
}

// WithSigV4Region sets the AWS region
func WithSigV4Region(region string) SigV4Option {
	return func(s *SigV4Authenticator) error {
		if region == "" {
			return &InvalidError{Type: "region"}
		}

		s.region = region
		return nil
	}
}

func newInitialAuthRequest(u string, region string, cred *credentials.Credentials) (*http.Request, error) {
	// Create a new HTTP request.
	req, err := http.NewRequest("GET", u, nil)
	if err != nil {
		return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
	}

	// Insert the query parameter response_type=assertion into the request.
	req.URL.RawQuery = url.Values{
		"response_type": {"assertion"},
	}.Encode()

	// Sign the request with SigV4.
	if _, err = sigv4.NewSigner(cred).Sign(req, nil, "execute-api", region, time.Now()); err != nil {
		return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
	}

	return req, nil
}

// NewSigV4Authenticator creates a session to initiate communication with
// Auth Server to obtain an access token using SigV4. Both region and credentials
// correspond to current AWS region and credentials for the SigV4 request.
//
// NOTE: By default, the call will use a HTTP client loaded with Amazon internal CA certs with a
//       60 second timeout. AWS region and credentials are default to environment variable settings
func NewSigV4Authenticator(options ...SigV4Option) (*SigV4Authenticator, error) {
	// Default values
	httpClient := amazoncacerts.GetHttpClient(false)
	httpClient.Timeout = 60 * time.Second

	sigV4Authenticator := &SigV4Authenticator{
		httpClient:  httpClient,
		region:      os.Getenv("AWS_REGION"),
		credentials: credentials.NewEnvCredentials(),
	}

	// Apply options
	for _, option := range options {
		if option != nil {
			if err := option(sigV4Authenticator); err != nil {
				return nil, err
			}
		}
	}

	// Follow redirection response.
	sigV4Authenticator.httpClient.CheckRedirect = func(req *http.Request, via []*http.Request) error {
		return http.ErrUseLastResponse
	}

	return sigV4Authenticator, nil
}

// Token generates a SigV4 request towards CloudAuth's Authorization Endpoint
// specified by url for authentication. If success, it returns an OAuth2 token.
func (s *SigV4Authenticator) Token(url string) (*oauth2.Token, error) {
	var authToken oauth2.Token

	// Generate and send a new request to Authorization Endpoint.
	req, err := newInitialAuthRequest(url, s.region, s.credentials)
	if err != nil {
		return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
	}

	r, err := s.httpClient.Do(req)
	if err != nil {
		return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
	}

	if r.StatusCode == http.StatusMovedPermanently {
		// Extract APIG's URL.
		loc, _ := r.Location()

		// Generate a new request towards CloudAuth's APIG.
		req, err := newInitialAuthRequest(loc.String(), s.region, s.credentials)
		if err != nil {
			return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
		}

		// Send request to Authorization Endpoint.
		r, err = s.httpClient.Do(req)
		if err != nil {
			return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
		}
	}

	if r.StatusCode != http.StatusOK {
		return nil, &GeneralError{fmt.Errorf("sigv4: Un-expected response code: %d", r.StatusCode)}
	}

	// Read the response body.
	body, err := ioutil.ReadAll(io.LimitReader(r.Body, maxResponseLen))
	r.Body.Close()
	if err != nil {
		return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
	}

	// Unmarshal the data.
	err = json.Unmarshal(body, &authToken)
	if err != nil {
		return nil, &GeneralError{fmt.Errorf("sigv4: %v", err)}
	}

	// Check for access_token.
	if authToken.AccessToken == "" {
		return nil, &GeneralError{fmt.Errorf("sigv4: missing access_token")}
	}

	return &authToken, nil
}
