package credentials

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
)

// This part is borrowed from Amazon Go SDK
//
// A Renewable is the interface for any component which will provide credentials
// Value. A creds is required to manage its own Expired state, and what to
// be expired means.
type Renewable interface {
	// Error is returned if the value were not obtainable, or empty.
	Get() (Value, error)
}

// A Value is the AWS credentials value for individual credential fields.
type Value struct {
	// AWS Access key ID
	AccessKeyID string

	// AWS Secret Access Key
	SecretAccessKey string

	// AWS Session Token
	SessionToken string
}

type RenewableCredentials struct {
	creds *credentials.Credentials
}

func NewRenewable(cfgs ...*aws.Config) (*RenewableCredentials, error) {
	sess, err := session.NewSession(cfgs...)
	if err != nil {
		return nil, err
	}
	return &RenewableCredentials{creds: sess.Config.Credentials}, nil
}

func Must(creds *RenewableCredentials, err error) *RenewableCredentials {
	if err != nil {
		panic(err)
	}
	return creds
}

func (r *RenewableCredentials) Get() (Value, error) {
	val, err := r.creds.Get()
	if err != nil {
		return Value{}, err
	}
	return Value{
		AccessKeyID:     val.AccessKeyID,
		SecretAccessKey: val.SecretAccessKey,
		SessionToken:    val.SessionToken,
	}, nil
}
