package brazilpath

import (
	"bytes"
	"fmt"
	"os"
	"os/exec"
	"strings"
)

// Graph represents the graph that is to be traversed when
// executing a recipe
type Graph string

type recipe string

// Recipe represents a brazil-path recipe that will be ran/cooked
// and split the output based on the delimiter provided.
type Recipe struct {
	recipe recipe
	delim  string
}

// NewRecipe will return a new Recipe specific to the recipe provided.
// When Cook is called delim is used to separate multiple entries
func NewRecipe(r, delim string) Recipe {
	return Recipe{
		recipe: recipe(r),
		delim:  delim,
	}
}

// Cook will perform the brazil-path command and split on the delimiter
// specified in the Recipe
func (r Recipe) Cook(g Graph) ([]string, error) {
	return r.recipe.runDelimited(g, r.delim)
}

// Graph enums
const (
	All        Graph = Graph("all")
	Build      Graph = Graph("build")
	Direct     Graph = Graph("direct")
	Lib        Graph = Graph("lib")
	Pkg        Graph = Graph("pkg")
	Run        Graph = Graph("run")
	TestBuild  Graph = Graph("testbuild")
	TestLib    Graph = Graph("testlib")
	TestRun    Graph = Graph("testrun")
	ToolDirect Graph = Graph("tooldirect")
	Tool       Graph = Graph("tool")
)

// Recipes
const (
	bin               recipe = recipe("bin")
	classpath         recipe = recipe("classpath")
	config            recipe = recipe("config")
	configFarm        recipe = recipe("configfarm")
	coralModel        recipe = recipe("coralmodel")
	gempath           recipe = recipe("gempath")
	generatedSrc      recipe = recipe("generatedsrc")
	graphviz          recipe = recipe("graphviz")
	includedirs       recipe = recipe("includedirs")
	jdk               recipe = recipe("jdk")
	libFarm           recipe = recipe("libfarm")
	lib               recipe = recipe("lib")
	linkLibs          recipe = recipe("linklibs")
	name              recipe = recipe("name")
	nameFullVersion   recipe = recipe("nameFullVersion")
	nameMajorVersion  recipe = recipe("nameMajorVersion")
	perlLib           recipe = recipe("perllib")
	pythonPath        recipe = recipe("pythonpath")
	relativeClasspath recipe = recipe("relativeclasspath")
	rubyLib           recipe = recipe("rubylib")
	src               recipe = recipe("src")
)

//PackageName returns the name of the current package
//It is equivalent to brazil-path package-name
func PackageName() (string, error) {
	return run(cmd("package-name"))
}

//PackageBuildRoot returns the path to the build directory of the current package
//It is equivalent to brazil-path package-build-root
func PackageBuildRoot() (string, error) {
	return run(cmd("package-build-root"))
}

//PackageSrcRoot returns the path to the current package
//It is equivalent to brazil-path package-src-root
func PackageSrcRoot() (string, error) {
	return run(cmd("package-src-root"))
}

// WithPackage will return a package specific graph
func (g Graph) WithPackage(p string) Graph {
	return Graph(fmt.Sprintf("[%s]%s", p, string(g)))
}

func (g Graph) CoralModel() ([]string, error) {
	return coralModel.runDelimited(g, " ")
}

func (g Graph) Name() ([]string, error) {
	return name.runDelimited(g, " ")
}

func (g Graph) Bin() ([]string, error) {
	return bin.runDelimited(g, ":")
}

func (g Graph) Lib() ([]string, error) {
	return lib.runDelimited(g, ":")
}

func cmd(param string) *exec.Cmd {
	return exec.Command("brazil-path", param)
}

//Run the command specified by the graph and recipe combination and return it's result
//separated by the delimiter
func (r recipe) runDelimited(g Graph, delimiter string) ([]string, error) {
	output, err := run(r.cmd(g))
	if err != nil {
		return nil, err
	}
	slice := strings.Split(output, delimiter)
	if len(slice) == 1 && slice[0] == "" {
		return []string{}, nil
	}
	return slice, nil
}

//Run the command specified by the graph and recipe combination and return it's result
//without leading and trailing whitespace
func (r recipe) runTrimmed(g Graph) (string, error) {
	return run(r.cmd(g))
}

func (r recipe) cmd(g Graph) *exec.Cmd {
	return cmd(string(g) + "." + string(r))
}

func run(cmd *exec.Cmd) (string, error) {
	stderr := bytes.Buffer{}
	cmd.Stderr = &stderr
	output, err := cmd.Output()
	if err != nil {
		fmt.Fprintf(os.Stderr, "$ %s %s\n", cmd.Path, strings.Join(cmd.Args[1:], " "))
		stderr.WriteTo(os.Stderr)
		return "", err
	}
	str := strings.TrimSpace(string(output))
	return str, nil
}
