/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package log

import (
	"GoFileRotate/rotate"
	"GoLog/log/level"
	"errors"
	"fmt"
	"github.com/tevino/abool"
	"io"
	"os"
	"path/filepath"
	"runtime"
	"strings"
	"sync"
	"time"
)

const (
	defaultBufferSize          = 100
	defaultDateFormat          = "Mon Jan 02 15:04:05 2006 UTC"
	defaultDepth               = -1
	defaultFlushDuration       = 1 * time.Second
	defaultStopTimeout         = 1 * time.Second
	defaultLogLevel            = level.Debug
	defaultPrefix              = "GoLog/log."
	defaultSrcSeparator        = string(filepath.Separator) + "src" + string(filepath.Separator)
	defaultDiscardOnBufferFull = false
)

var (
	// Make it possible to test exiting on FATAL.
	osExit = os.Exit
)

type AmznLogger struct {
	depth               int
	exitOnFatal         bool
	defaultLevel        level.Level
	writer              io.Writer
	packageLevels       map[string]level.Level
	applicationName     string
	dateFormat          string
	hostname            string
	prefix              string
	stopTimeout         time.Duration
	flushChan           <-chan time.Time
	stopChan            chan bool
	isRunning           *abool.AtomicBool
	writeChan           chan logRecord
	generateLogFunc     func(record logRecord) string
	discardOnBufferFull bool
	closeOnce           sync.Once
}

type FormatOptions struct {
	SkipAppName  bool
	SkipPid      bool
	SkipHostname bool
	SkipFuncName bool
}

// SkipCount returns number of skipped components.
func (fo FormatOptions) SkipCount() int {
	count := 0
	if fo.SkipAppName {
		count++
	}
	if fo.SkipPid {
		count++
	}
	if fo.SkipHostname {
		count++
	}
	if fo.SkipFuncName {
		count++
	}
	return count
}

type logRecord struct {
	date     time.Time
	line     int
	pid      int
	level    level.Level
	buff     string
	caller   string
	funcName string
}

// Used to enable AmznLogger to flush buffers of the writer if it exposes a Flush() function, such as bufio.Writer.
// Buffers must be flushed to ensure all logs get written out.
type Flusher interface {
	Flush() error
}

// Starts a new Amazon Logger instance that generates logs in the standard Amazon application log format.
// The logger writes to a specified io.Writer interface such as the console, or a file.
// Any number of unique options may be passed in to configure the AmznLogger instance.
func NewAmznLogger(writer io.Writer, applicationName string, options ...func(*AmznLogger) error) (*AmznLogger, error) {
	if writer == nil {
		return nil, errors.New("writer must be specified.")
	}
	hostname, _ := os.Hostname()
	l := &AmznLogger{
		writer:              writer,
		packageLevels:       make(map[string]level.Level),
		applicationName:     applicationName,
		defaultLevel:        defaultLogLevel,
		hostname:            hostname,
		dateFormat:          defaultDateFormat,
		prefix:              defaultPrefix,
		depth:               defaultDepth,
		stopTimeout:         defaultStopTimeout,
		flushChan:           make(chan time.Time),
		writeChan:           make(chan logRecord, defaultBufferSize),
		stopChan:            make(chan bool),
		discardOnBufferFull: defaultDiscardOnBufferFull,
		isRunning:           abool.NewBool(true),
	}
	l.generateLogFunc = func(record logRecord) string {
		return fmt.Sprintf("%v %v %v-0@%v:0 %v %v:%v %v(): %v", record.date.Format(l.dateFormat),
			applicationName, record.pid, hostname, level.LevelStr(record.level), record.caller, record.line,
			record.funcName, record.buff)
	}

	for _, option := range options {
		err := option(l)
		if err != nil {
			return nil, fmt.Errorf("Error creating AmznLogger. Error while setting options: %v", err)
		}
	}
	// Start Go Routine that writes out logs.
	go l.writeChannelProcessor()
	return l, nil
}

// Constructor option to set the default LogLevel for all packages.
func LogLevel(level level.Level) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		l.defaultLevel = level
		return nil
	}
}

// Constructor option to turn on/off whether or not a FATAL log will
// result in os.Exit being called.
func ExitOnFatal(exit bool) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		l.exitOnFatal = exit
		return nil
	}
}

// Constructor option to set a custom date/time format.
func DateFormat(format string) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		l.dateFormat = format
		return nil
	}
}

// SetFormatOptions sets the format for each log line. By default, a log line looks like below:
// "date appName pid-0@hostname:0 level caller:line funcName(): msg"
func SetFormatOptions(formatOptions FormatOptions) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		numComponents := 9 // date appName pid hostname level caller line functionName msg

		// Generate format string to log generation.
		format := "%v" // date

		if formatOptions.SkipAppName == false {
			format += " %v" // appName
		}
		if formatOptions.SkipPid == false {
			format += " %v-0" // pid
		}
		if formatOptions.SkipHostname == false {
			if formatOptions.SkipPid {
				format += " "
			}
			format += "@%v:0" // hostname
		}
		format += " %v"    // level
		format += " %v:%v" // caller:line
		if formatOptions.SkipFuncName == false {
			format += " %v():" // funcName
		}
		format += " %v" // msg

		l.generateLogFunc = func(record logRecord) string {
			args := make([]interface{}, numComponents-formatOptions.SkipCount())
			i := 0
			args[i], i = record.date.Format(l.dateFormat), i+1
			if formatOptions.SkipAppName == false {
				args[i], i = l.applicationName, i+1
			}
			if formatOptions.SkipPid == false {
				args[i], i = record.pid, i+1
			}
			if formatOptions.SkipHostname == false {
				args[i], i = l.hostname, i+1
			}
			args[i], i = level.LevelStr(record.level), i+1
			args[i], i = record.caller, i+1
			args[i], i = record.line, i+1
			if formatOptions.SkipFuncName == false {
				args[i], i = record.funcName, i+1
			}
			args[i], i = record.buff, i+1
			return fmt.Sprintf(format, args...)
		}
		return nil
	}
}

// Constructor option to specify the size of the internal buffer.
// Default buffer size is 100 log messages.
func BufferSize(size int) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		if size < 0 {
			return errors.New("BufferSize must be >= 0.")
		}
		l.writeChan = make(chan logRecord, size)
		return nil
	}
}

// Constructor option to specify whether to discard write operations
// if the logger's buffer is full.
func DiscardOnBufferFull(discard bool) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		l.discardOnBufferFull = discard
		return nil
	}
}

// Constructor option to specify a package prefix for determining callstack depth.
// This should only be used when using a proxy in conjunction with the AmznLogger.
func LoggerPackagePrefix(prefix string) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		if len(prefix) == 0 {
			return errors.New("PackagePrefix length must be > 0!")
		}
		l.prefix = prefix
		return nil
	}
}

// Constructor option to specify the timeout when stopping the logger.
// Input time is time.Duration and must be >= 0. The default timeout is 1 second.
func StopTimeout(t time.Duration) func(*AmznLogger) error {
	return func(l *AmznLogger) error {
		if t < 0 {
			return fmt.Errorf("Stop Timeout must be >= 0. Specified timeout: %v", t)
		}
		l.stopTimeout = t
		return nil
	}
}

// Sets the specified package to the specified log level.
func (l *AmznLogger) AddPackageLogLevel(pack string, level level.Level) {
	l.packageLevels[pack] = level
}

// Sets the specified packages to the specified log levels.
func (l *AmznLogger) AddPackageLogLevelMap(m map[string]level.Level) {
	for k, v := range m {
		l.AddPackageLogLevel(k, v)
	}
}

// Gets the package log level for the specified package.
func (l *AmznLogger) GetPackageLogLevel(pack string) *level.Level {
	if val, ok := l.packageLevels[pack]; ok {
		return &val
	}
	return nil
}

// Get the package log levels for the specified packages.
func (l *AmznLogger) GetPackageLogLevelMap(packs []string) map[string]*level.Level {
	result := make(map[string]*level.Level, len(packs))
	for _, val := range packs {
		result[val] = l.GetPackageLogLevel(val)
	}
	return result
}

// Removes the log level from the specified package.
func (l *AmznLogger) RemovePackageLogLevel(pack string) {
	delete(l.packageLevels, pack)
}

// Remotes the log levels from the specified packages.
func (l *AmznLogger) RemovePackageLogLevelSlice(packs []string) {
	for _, val := range packs {
		l.RemovePackageLogLevel(val)
	}
}

func (l *AmznLogger) writeChannelProcessor() {
	flushable, _ := l.writer.(Flusher)
	if flushable != nil {
		// Writer is flushable. Ensure buffer is flushed when logger exits.
		defer flushable.Flush()
		l.flushChan = time.Tick(defaultFlushDuration)
	}
	for {
		select {
		case record := <-l.writeChan:
			l.writeLog(record)
		case <-l.stopChan:
			// close the writeChan, then dump all logs to the logger. If the writer is flushable, the defered flush will flush it.
			close(l.writeChan)
			for r := range l.writeChan {
				l.writeLog(r)
			}
			return
		case <-l.flushChan:
			flushable.Flush()
		}
	}
}

func (l *AmznLogger) writeLog(record logRecord) {
	str := l.generateLogFunc(record)
	switch t := l.writer.(type) {
	case rotate.TemporalWriter:
		t.TimedWrite([]byte(str), &record.date)
	case io.Writer:
		t.Write([]byte(str))
	}
}

func (l *AmznLogger) enqueueLog(message string, level level.Level, file string, line int, funcName string) {
	if l.Running() {
		// If writeChan has been closed recover from the panic and continue. Unlikely to happen, but protecting against corner cases.
		defer func() {
			if r := recover(); r != nil {
				os.Stderr.Write([]byte("[ERROR] Unable to log message as logger has been stopped. Panic recovered."))
			}
		}()
		pid := os.Getpid()
		record := logRecord{
			date:     time.Now().UTC(),
			line:     line,
			pid:      pid,
			level:    level,
			buff:     message,
			caller:   file,
			funcName: funcName,
		}
		if l.discardOnBufferFull {
			select {
			case l.writeChan <- record:
			default: // discard record
			}
			return
		}
		l.writeChan <- record
	} else {
		os.Stderr.Write([]byte("[ERROR] Unable to log message as logger has been stopped!"))
	}
}

// Gets the LogLevel for the specified function name.
// Checks if the package of the function has a specified LogLevel. It does this by trimming off the last '.' to isolate the package
// then trims on '/' to isolate package levels. Log level is determined by trimming off package levels until it finds the most specific
// package with a custom logging level. If all package levels are trimming and no level is found, it defaults to the specified default logging level.
// Example:
// For funcName "go.amzn.com/foo/bar/baz.MyFunc", we would look for "go.amzn.com/foo/bar/baz", then "go.amzn.com/foo/bar", then "go.amzn.com/foo",
// and finally "go.amzn.com" before returning the default log level.
func (l *AmznLogger) getLogLevel(funcName string) level.Level {
	// Bail out early if we don't have any packageLevels defined
	if len(l.packageLevels) == 0 {
		return l.defaultLevel
	}

	// Strip off the last .foobar
	if lastDot := strings.LastIndex(funcName, "."); lastDot != -1 {
		funcName = funcName[:lastDot]
	}
	// Now strip off package levels from the end until we run out of packages to check
	for len(funcName) > 0 {
		if val, ok := l.packageLevels[funcName]; ok {
			return val
		}

		if lastSlash := strings.LastIndex(funcName, "/"); lastSlash != -1 {
			if lastDot := strings.LastIndex(funcName, "."); lastDot > lastSlash {
				funcName = funcName[:lastDot]
			} else {
				funcName = funcName[:lastSlash]
			}
		} else {
			break
		}
	}
	return l.defaultLevel
}

// Log a message using the default format for the provided values.  level specifies the log level for the provided
// message.  callDepth specifies the number of stack frames to skip when determining the calling function name to
// include in the log message.  If callDepth is negative the default behavior is used.
func (l *AmznLogger) Logln(callDepth int, level level.Level, v ...interface{}) {
	file, line, funcName := l.getFileLineFuncName(callDepth)
	minLevel := l.getLogLevel(funcName)
	if minLevel > level {
		return
	}
	l.enqueueLog(fmt.Sprintln(v...), level, file, line, funcName)
	if l.exitOnFatal {
		l.fatalCheck(level)
	}
}

// Log a message formatting the values using the specified format string.  callDepth and level are
// interpreted in the same manner as Logln.
func (l *AmznLogger) Logf(callDepth int, level level.Level, format string, v ...interface{}) {
	file, line, funcName := l.getFileLineFuncName(callDepth)
	minLevel := l.getLogLevel(funcName)
	if minLevel > level {
		return
	}
	if format[len(format)-1] != '\n' {
		format += "\n"
	}
	l.enqueueLog(fmt.Sprintf(format, v...), level, file, line, funcName)
	if l.exitOnFatal {
		l.fatalCheck(level)
	}
}

// Log a message using the default format for the provided values with ancillary key/value fields.  callDepth and
// level are interpreted in the same manner as Logln.
func (l *AmznLogger) Logw(callDepth int, level level.Level, fields Fields, v ...interface{}) {
	file, line, funcName := l.getFileLineFuncName(callDepth)
	minLevel := l.getLogLevel(funcName)
	if minLevel > level {
		return
	}
	var sb strings.Builder
	for k, v := range fields {
		fmt.Fprintf(&sb, "%s=%v ", k, v)
	}
	l.enqueueLog(sb.String()+fmt.Sprintln(v...), level, file, line, funcName)
	if l.exitOnFatal {
		l.fatalCheck(level)
	}
}

// fatalCheck inspects the given level, and if it's a FATAL will close the logger and
// exit the program.  fatalCheck should only be called when exitOnFatal is set to true.
// Checking the exitOnFatal flag is done outside this function to minimize the impact
// on applications that keep it disabled.
func (l *AmznLogger) fatalCheck(lvl level.Level) {
	if lvl == level.Fatal {
		l.Close()
		osExit(1)
	}
}

func (l *AmznLogger) Close() error {
	// Will only try to close the logger once, will return within the configured timeout
	var err error
	l.closeOnce.Do(func() {
		c := make(chan bool, 1)
		go func() {
			// Unset makes boolean false
			l.isRunning.UnSet()
			l.stopChan <- true
			if closer, ok := l.writer.(io.Closer); ok {
				closer.Close()
			}
			c <- true
		}()
		select {
		case <-c:
			return
		case <-time.After(l.stopTimeout):
			err = fmt.Errorf("failed to Close logger. Timed out after %v", l.stopTimeout)
			return
		}
	})
	return err
}

// Checks to see if this logger is actively logging, or has been stopped.
// Return true if actively logging, false if it been stopped.
func (l *AmznLogger) Running() bool {
	return l.isRunning.IsSet()
}

func (l *AmznLogger) Trace(v ...interface{}) {
	l.Logln(defaultDepth, level.Trace, v...)
}

func (l *AmznLogger) Tracew(fields Fields, v ...interface{}) {
	l.Logw(defaultDepth, level.Trace, fields, v...)
}

func (l *AmznLogger) Tracef(format string, v ...interface{}) {
	l.Logf(defaultDepth, level.Trace, format, v...)
}

func (l *AmznLogger) Debug(v ...interface{}) {
	l.Logln(defaultDepth, level.Debug, v...)
}

func (l *AmznLogger) Debugf(format string, v ...interface{}) {
	l.Logf(defaultDepth, level.Debug, format, v...)
}

func (l *AmznLogger) Debugw(fields Fields, v ...interface{}) {
	l.Logw(defaultDepth, level.Debug, fields, v...)
}

func (l *AmznLogger) Info(v ...interface{}) {
	l.Logln(defaultDepth, level.Info, v...)
}

func (l *AmznLogger) Infof(format string, v ...interface{}) {
	l.Logf(defaultDepth, level.Info, format, v...)
}

func (l *AmznLogger) Infow(fields Fields, v ...interface{}) {
	l.Logw(defaultDepth, level.Info, fields, v...)
}

func (l *AmznLogger) Warn(v ...interface{}) {
	l.Logln(defaultDepth, level.Warn, v...)
}

func (l *AmznLogger) Warnf(format string, v ...interface{}) {
	l.Logf(defaultDepth, level.Warn, format, v...)
}

func (l *AmznLogger) Warnw(fields Fields, v ...interface{}) {
	l.Logw(defaultDepth, level.Warn, fields, v...)
}

func (l *AmznLogger) Error(v ...interface{}) {
	l.Logln(defaultDepth, level.Error, v...)
}

func (l *AmznLogger) Errorf(format string, v ...interface{}) {
	l.Logf(defaultDepth, level.Error, format, v...)
}

func (l *AmznLogger) Errorw(fields Fields, v ...interface{}) {
	l.Logw(defaultDepth, level.Error, fields, v...)
}

func (l *AmznLogger) Fatal(v ...interface{}) {
	l.Logln(defaultDepth, level.Fatal, v...)
}

func (l *AmznLogger) Fatalf(format string, v ...interface{}) {
	l.Logf(defaultDepth, level.Fatal, format, v...)
}

func (l *AmznLogger) Fatalw(fields Fields, v ...interface{}) {
	l.Logw(defaultDepth, level.Fatal, fields, v...)
}

// Gets the file path, line number, and function name of where the log statement was called.  callDepth specifies
// the number of stack frames to skip to obtain the invoking caller information.  If callDepth is negative the
// default behavior of traversing the stack and finding the provided prefix is used.
func (l *AmznLogger) getFileLineFuncName(callDepth int) (file string, line int, funcName string) {
	if callDepth >= 0 {
		// Increment the user provided callDepth by two frames.  One for this function, and the second for
		// the Logf/Logln/Logw functions.
		callDepth += 2
	} else {
		if l.depth == defaultDepth {
			// Identify what callstack callDepth to retrieve file, line number, and function name.
			// Only runs the first time something is logged.
			seenPrefix := false
			for i := 1; true; i++ {
				pc, _, _, _ := runtime.Caller(i)
				funcName = runtime.FuncForPC(pc).Name()
				if funcName == "" {
					// Reached top of callstack and callDepth not found. Will return blank path, line number, and function name.
					// This likely indicates the specified prefix does not exist in the callstack.
					l.depth = i
					break
				}
				// Iterate up the call stack until we find the first occurence of the prefix.
				// Continue iterating until we no longer find the prefix and return.
				hasPrefix := strings.HasPrefix(funcName, l.prefix)
				if seenPrefix && !hasPrefix {
					l.depth = i
					break
				}
				if !seenPrefix && hasPrefix {
					seenPrefix = true
				}
			}
		}
		callDepth = l.depth
	}
	pc, file, line, _ := runtime.Caller(callDepth)
	funcName = runtime.FuncForPC(pc).Name()
	if p := strings.LastIndex(file, defaultSrcSeparator); p >= 0 {
		return file[p+len(defaultSrcSeparator):], line, funcName
	}
	return file, line, funcName
}
