package log

import (
	"GoLog/log/level"
	"bytes"
	"errors"
	"fmt"
	"github.com/tevino/abool"
	"io"
	"os"
	"regexp"
	"strconv"
	"strings"
	"sync"
	"sync/atomic"
	"testing"
	"time"
)

type TestWriter struct {
	buffer      bytes.Buffer
	bufferMutex sync.Mutex
	flushed     *abool.AtomicBool
	r           chan bool
}

func NewTestWriter() *TestWriter {
	writer := TestWriter{
		buffer:  bytes.Buffer{},
		flushed: abool.NewBool(false),
		r:       make(chan bool, 1),
	}
	return &writer
}

func (t *TestWriter) Write(v []byte) (n int, err error) {
	t.bufferMutex.Lock()
	n, err = t.buffer.Write(v)
	t.bufferMutex.Unlock()
	t.r <- true
	return n, err
}

func (t *TestWriter) Flush() error {
	t.flushed.Set()
	return nil
}

func (t *TestWriter) Flushed() bool {
	return t.flushed.IsSet()
}

func (t *TestWriter) String() string {
	t.bufferMutex.Lock()
	defer t.bufferMutex.Unlock()
	return t.buffer.String()
}

const (
	defaultTimeout  = 1 * time.Second
	defaultAppName  = "Tester"
	defaultHostName = "host"
	testString1     = "Test String 1"
)

var (
	defaultDate      = time.Now().UTC()
	defaultDateStr   = defaultDate.Format(defaultDateFormat)
	defaultLogRecord = logRecord{
		date:     defaultDate,
		line:     1,
		pid:      3,
		level:    level.Info,
		buff:     "msg",
		caller:   "caller",
		funcName: "func",
	}
	testFormat    = "Test Format %s"
	testFormatVal = "Test"
	testFieldsStr = "key=val"
)

var (
	testFields = Fields{"key": "val"}

	allLevels = []level.Level{level.Trace, level.Debug, level.Info, level.Warn, level.Error, level.Fatal}

	testRegexStdTrace = regexp.MustCompile(`\[TRACE\].*` + testString1)
	testRegexStdDebug = regexp.MustCompile(`\[DEBUG\].*` + testString1)
	testRegexStdInfo  = regexp.MustCompile(`\[INFO\].*` + testString1)
	testRegexStdWarn  = regexp.MustCompile(`\[WARN\].*` + testString1)
	testRegexStdError = regexp.MustCompile(`\[ERROR\].*` + testString1)
	testRegexStdFatal = regexp.MustCompile(`\[FATAL\].*` + testString1)

	logRegexes = []*regexp.Regexp{testRegexStdTrace, testRegexStdDebug, testRegexStdInfo, testRegexStdWarn,
		testRegexStdError, testRegexStdFatal}

	expectedFormatStr  = fmt.Sprintf(testFormat, testFormatVal)
	testRegexStdTracef = regexp.MustCompile(`\[TRACE\].*` + expectedFormatStr)
	testRegexStdDebugf = regexp.MustCompile(`\[DEBUG\].*` + expectedFormatStr)
	testRegexStdInfof  = regexp.MustCompile(`\[INFO\].*` + expectedFormatStr)
	testRegexStdWarnf  = regexp.MustCompile(`\[WARN\].*` + expectedFormatStr)
	testRegexStdErrorf = regexp.MustCompile(`\[ERROR\].*` + expectedFormatStr)
	testRegexStdFatalf = regexp.MustCompile(`\[FATAL\].*` + expectedFormatStr)

	logfRegexes = []*regexp.Regexp{testRegexStdTracef, testRegexStdDebugf, testRegexStdInfof, testRegexStdWarnf,
		testRegexStdErrorf, testRegexStdFatalf}

	expectedStrWithFields = fmt.Sprintf("%s %s", testFieldsStr, testString1)
	testRegexStdTracew    = regexp.MustCompile(`\[TRACE\].*` + expectedStrWithFields)
	testRegexStdDebugw    = regexp.MustCompile(`\[DEBUG\].*` + expectedStrWithFields)
	testRegexStdInfow     = regexp.MustCompile(`\[INFO\].*` + expectedStrWithFields)
	testRegexStdWarnw     = regexp.MustCompile(`\[WARN\].*` + expectedStrWithFields)
	testRegexStdErrorw    = regexp.MustCompile(`\[ERROR\].*` + expectedStrWithFields)
	testRegexStdFatalw    = regexp.MustCompile(`\[FATAL\].*` + expectedStrWithFields)

	logwRegexes = []*regexp.Regexp{testRegexStdTracew, testRegexStdDebugw, testRegexStdInfow, testRegexStdWarnw,
		testRegexStdErrorw, testRegexStdFatalw}
)

func logGenerator(logger *AmznLogger) {
	logger.Trace(testString1)
	logger.Debug(testString1)
	logger.Info(testString1)
	logger.Warn(testString1)
	logger.Error(testString1)
	logger.Fatal(testString1)
}

func logfGenerator(logger *AmznLogger) {
	logger.Tracef(testFormat, testFormatVal)
	logger.Debugf(testFormat, testFormatVal)
	logger.Infof(testFormat, testFormatVal)
	logger.Warnf(testFormat, testFormatVal)
	logger.Errorf(testFormat, testFormatVal)
	logger.Fatalf(testFormat, testFormatVal)
}

func logwGenerator(logger *AmznLogger) {
	logger.Tracew(testFields, testString1)
	logger.Debugw(testFields, testString1)
	logger.Infow(testFields, testString1)
	logger.Warnw(testFields, testString1)
	logger.Errorw(testFields, testString1)
	logger.Fatalw(testFields, testString1)
}

func TestNewAmznLogger_NilWriterError(t *testing.T) {
	_, err := NewAmznLogger(nil, defaultAppName)
	if err == nil {
		t.Fatal("Expected an error, however none was returned!")
	}
}

func TestNewAmznLogger_OptionsReturnError(t *testing.T) {
	errorOption := func(*AmznLogger) error {
		return errors.New("Expected")
	}
	_, err := NewAmznLogger(NewTestWriter(), defaultAppName, errorOption)
	if err == nil {
		t.Fatal("Expected an error, however none was returned!")
	}
}

func TestNewAmznLogger_NoOptionsError(t *testing.T) {
	option := func(*AmznLogger) error { return nil }
	_, err := NewAmznLogger(NewTestWriter(), defaultAppName, option)
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
}

func TestNewAmznLogger_ExitOnFatal_Setup(t *testing.T) {
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, ExitOnFatal(true))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
	if logger.exitOnFatal != true {
		t.Fatal("Expected exitOnFatal: true. Actual log level was: false.")
	}
}

func TestNewAmznLogger_ExitOnFatal_ExitCalled(t *testing.T) {
	origExit := osExit
	defer func() { osExit = origExit }()

	exitCalled := false
	osExit = func(int) { exitCalled = true }

	// The default is to not exit on a Fatal log, so other tests cover
	// the non-exiting case.
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName, ExitOnFatal(true))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	logger.Fatal("true")
	if !exitCalled {
		t.Fatal("Exit was not called when a FATAL was logged.")
	}
}

func TestNewAmznLogger_ExitOnFatalf_ExitCalled(t *testing.T) {
	origExit := osExit
	defer func() { osExit = origExit }()

	exitCalled := false
	osExit = func(int) { exitCalled = true }

	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName, ExitOnFatal(true))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	logger.Fatalf("%v", true)
	if !exitCalled {
		t.Fatal("Exit was not called when a formatted FATAL was logged.")
	}
}

func TestNewAmznLogger_DateFormat(t *testing.T) {
	format := "this is a test format"
	waitGroup := sync.WaitGroup{}
	waitGroup.Add(1)
	writer := &slowWriter{delay: 0, logs: []string{}, waitGroup: &waitGroup}
	logger, err := NewAmznLogger(writer, defaultAppName, DateFormat(format))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
	if logger.dateFormat != format {
		t.Fatalf("Expected dateFormat: %v. Actual dateFormat was: %v.", format, logger.dateFormat)
	}
	logger.Info("Test date format")

	waitGroup.Wait()
	logs := writer.logs
	if len(logs) != 1 {
		t.Fatalf("Expected 1 log entry, found: %d", len(logs))
	}
	entry := logs[0]
	if !strings.HasPrefix(entry, format) {
		t.Fatalf("Expected [%s] format to be present in log entry [%s]", format, entry)
	}
}

func TestLogFormat(t *testing.T) {
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName)
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}

	date := time.Now().UTC()
	dateStr := date.Format(defaultDateFormat)
	record := logRecord{
		date:     date,
		line:     1,
		pid:      3,
		level:    level.Info,
		buff:     "msg",
		caller:   "caller",
		funcName: "func",
	}

	hostname, _ := os.Hostname()
	expectedLogStr := fmt.Sprintf("%s %s %d-0@%s:0 %s %s:%d %s(): %s", dateStr, defaultAppName, record.pid,
		hostname, level.LevelStr(record.level), record.caller, record.line, record.funcName, record.buff)

	logStr := logger.generateLogFunc(record)
	if logStr != expectedLogStr {
		t.Fatalf("Expected log: %v. Actual log: %v.", expectedLogStr, logStr)
	}
}

func TestLogFormatEmptyFormatOptions(t *testing.T) {
	options := FormatOptions{}
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, SetFormatOptions(options))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
	logger.hostname = defaultHostName

	expectedLogStr := fmt.Sprintf("%s %s %d-0@%s:0 %s %s:%d %s(): %s", defaultDateStr, defaultAppName,
		defaultLogRecord.pid, defaultHostName, level.LevelStr(defaultLogRecord.level), defaultLogRecord.caller,
		defaultLogRecord.line, defaultLogRecord.funcName, defaultLogRecord.buff)

	logStr := logger.generateLogFunc(defaultLogRecord)
	if logStr != expectedLogStr {
		t.Fatalf("Expected log: %v. Actual log: %v.", expectedLogStr, logStr)
	}
}

func TestLogFormat_SkipAppName(t *testing.T) {
	options := FormatOptions{
		SkipAppName: true,
	}
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, SetFormatOptions(options))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
	logger.hostname = defaultHostName

	expectedLogStr := fmt.Sprintf("%s %d-0@%s:0 %s %s:%d %s(): %s", defaultDateStr, defaultLogRecord.pid,
		defaultHostName, level.LevelStr(defaultLogRecord.level), defaultLogRecord.caller, defaultLogRecord.line,
		defaultLogRecord.funcName, defaultLogRecord.buff)

	logStr := logger.generateLogFunc(defaultLogRecord)
	if logStr != expectedLogStr {
		t.Fatalf("Expected log: %v. Actual log: %v.", expectedLogStr, logStr)
	}
}

func TestLogFormat_SkipPid(t *testing.T) {
	options := FormatOptions{
		SkipPid: true,
	}
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, SetFormatOptions(options))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
	logger.hostname = defaultHostName

	expectedLogStr := fmt.Sprintf("%s %s @%s:0 %s %s:%d %s(): %s", defaultDateStr, defaultAppName,
		defaultHostName, level.LevelStr(defaultLogRecord.level), defaultLogRecord.caller, defaultLogRecord.line,
		defaultLogRecord.funcName, defaultLogRecord.buff)

	logStr := logger.generateLogFunc(defaultLogRecord)
	if logStr != expectedLogStr {
		t.Fatalf("Expected log: %v. Actual log: %v.", expectedLogStr, logStr)
	}
}

func TestLogFormat_SkipHostName(t *testing.T) {
	options := FormatOptions{
		SkipHostname: true,
	}
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, SetFormatOptions(options))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}

	expectedLogStr := fmt.Sprintf("%s %s %d-0 %s %s:%d %s(): %s", defaultDateStr, defaultAppName,
		defaultLogRecord.pid, level.LevelStr(defaultLogRecord.level), defaultLogRecord.caller,
		defaultLogRecord.line, defaultLogRecord.funcName, defaultLogRecord.buff)

	logStr := logger.generateLogFunc(defaultLogRecord)
	if logStr != expectedLogStr {
		t.Fatalf("Expected log: %v. Actual log: %v.", expectedLogStr, logStr)
	}
}

func TestLogFormat_SkipFuncName(t *testing.T) {
	options := FormatOptions{
		SkipFuncName: true,
	}
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, SetFormatOptions(options))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
	logger.hostname = defaultHostName

	expectedLogStr := fmt.Sprintf("%s %s %d-0@%s:0 %s %s:%d %s", defaultDateStr, defaultAppName,
		defaultLogRecord.pid, defaultHostName, level.LevelStr(defaultLogRecord.level), defaultLogRecord.caller,
		defaultLogRecord.line, defaultLogRecord.buff)

	logStr := logger.generateLogFunc(defaultLogRecord)
	if logStr != expectedLogStr {
		t.Fatalf("Expected log: %v. Actual log: %v.", expectedLogStr, logStr)
	}
}

func TestNewAmznLogger_BufferSize_InvalidBufferSize(t *testing.T) {
	size := -1
	_, err := NewAmznLogger(NewTestWriter(), defaultAppName, BufferSize(size))
	if err == nil {
		t.Fatal("Expected an error, however none was returned!")
	}
}

func TestNewAmznLogger_BufferSize_ValidZeroBufferSize(t *testing.T) {
	size := 0
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, BufferSize(size))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v.", err)
	}
	if cap(logger.writeChan) != size {
		t.Fatalf("Expected writeChan size: %v. Actual writeChan size: %v.", size, logger.writeChan)
	}
}

func TestNewAmznLogger_BufferSize_ValidNonZeroBufferSize(t *testing.T) {
	size := 150
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, BufferSize(size))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v.", err)
	}
	if cap(logger.writeChan) != size {
		t.Fatalf("Expected writeChan size: %v. Actual writeChan size: %v.", size, logger.writeChan)
	}
}

func TestNewAmznLogger_LoggerPackagePrefix_InvalidPrefix(t *testing.T) {
	prefix := ""
	_, err := NewAmznLogger(NewTestWriter(), defaultAppName, LoggerPackagePrefix(prefix))
	if err == nil {
		t.Fatal("Expected an error, however none was returned!")
	}
}

func TestNewAmznLogger_LoggerPackagePrefix_ValidPrefix(t *testing.T) {
	prefix := "testPrefix"
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, LoggerPackagePrefix(prefix))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v.", err)
	}
	if logger.prefix != prefix {
		t.Fatalf("Expected prefix: %v. Actual prefix: %v.", prefix, logger.prefix)
	}
}

func TestNewAmznLogger_WriteChannelProcessor_Flush(t *testing.T) {
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName)
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v.", err)
	}
	logger.stopChan <- true
	timeout := time.After(defaultTimeout)
	<-timeout
	if w.Flushed() != true {
		t.Fatal("Writer was not flushed properly")
	}
}

func TestNewAmznLogger_WriteChannelProcessor_WriterWrite(t *testing.T) {
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName)
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	timeout := time.After(defaultTimeout)
	logger.Fatal(testString1)
	select {
	case <-w.r:
		result := w.String()
		if !strings.Contains(result, testString1) {
			t.Fatalf("Unexpected log output. expected log to contain: %v. Actual log: %v.", testString1, result)
		}
	case <-timeout:
		t.Fatal("Logger timed out!")
	}
}

func TestNewAmznLogger_Close(t *testing.T) {
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName)
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	// Close multiple times to ensure that no error is caused by doing so.
	if err := logger.Close(); err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	if err := logger.Close(); err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	// Pause. Wait for logger to exit. Should take much less than 10 ms.
	time.Sleep(10 * time.Millisecond)
	if logger.Running() {
		t.Fatal("Logger is still running even after being stopped.")
	}
}

func TestNewAmznLogger_CloseLogger_ThenWrite(t *testing.T) {
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName)
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}

	logger.Info("Writing before closing")
	c := make(chan bool, 1)
	go func() {
		if err := logger.Close(); err != nil {
			c <- false
		}
		c <- true
	}()

	logger.Info("Try to write concurrently with the Close")
	<-c
	logger.Info("Try to write after the closure of logger")

	logLines := strings.SplitAfter(strings.Trim(w.String(), "\n"), "\n")
	if len(logLines) > 2 {
		t.Fatalf("Expected at most 2 logs to be logged, found: [%d] containing: [%v]", len(logLines), logLines)
	}
}

func TestNewAmznLogger_Close_StopTimeout(t *testing.T) {
	closeTimeout := 10 * time.Microsecond
	closeCounter := int64(0)
	waitGroup := &sync.WaitGroup{}
	waitGroup.Add(1)

	w := &sleepOnCloseWriter{closeDelay: closeTimeout * 2, closeCounter: &closeCounter, waitGroup: waitGroup}
	logger, err := NewAmznLogger(w, defaultAppName, StopTimeout(closeTimeout))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}

	startTime := time.Now()
	err = logger.Close()
	if err == nil {
		t.Fatalf("Expected a timeout error")
	}
	expectedErrorStr := fmt.Sprintf("failed to Close logger. Timed out after %v", closeTimeout)
	if err.Error() != expectedErrorStr {
		t.Fatalf("Expected Error message to say: [%s], Found: [%s]", expectedErrorStr, err.Error())
	}

	if diff := time.Now().Sub(startTime); diff < closeTimeout {
		t.Fatalf("Expected Close to take atleast [%s] but it took only [%s]", closeTimeout, diff)
	}

	// Close again and make sure that close does not error out nor does not wait on close again
	startTime = time.Now()
	if err := logger.Close(); err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	if diff := time.Now().Sub(startTime); diff > closeTimeout {
		t.Fatalf("Expected Close to take less [%s] but it took [%s]", closeTimeout, diff)
	}

	waitGroup.Wait()
	// Verify that the close method is invoked exactly once
	if count := atomic.LoadInt64(&closeCounter); count != 1 {
		t.Fatalf("Expected Close to be invoked once but it was invoked [%d] times", count)
	}
}

type sleepOnCloseWriter struct {
	closeDelay   time.Duration
	closeCounter *int64
	waitGroup    *sync.WaitGroup
	io.Closer
}

func (w *sleepOnCloseWriter) Write(p []byte) (n int, err error) {
	return len(p), nil
}

func (w *sleepOnCloseWriter) Close() error {
	w.waitGroup.Done()
	atomic.AddInt64(w.closeCounter, 1)
	time.Sleep(w.closeDelay)
	return nil
}

func TestNewAmznLogger_LogLevel_Setup(t *testing.T) {
	level := level.Fatal
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, LogLevel(level))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}
	if logger.defaultLevel != level {
		t.Fatalf("Expected log level: %v. Actual log level was: %v.", level, logger.defaultLevel)
	}
}

func TestNewAmznLogger_LogLevel_IgnoreLowerLevelLogs(t *testing.T) {
	level := level.Error
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName, LogLevel(level))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	timeout := time.After(defaultTimeout)
	logger.Trace(testString1)
	select {
	case <-w.r:
		result := w.String()
		t.Fatalf("Unexpected log output. Expected log to contain: %v. Actual log: %v.", "", result)
	case <-timeout:
	}
}

func TestNewAmznLogger_LogLevel_ContainsSameLevelLogs(t *testing.T) {
	level := level.Error
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName, LogLevel(level))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	timeout := time.After(defaultTimeout)
	logger.Error(testString1)
	select {
	case <-w.r:
		result := w.String()
		if !strings.Contains(result, testString1) {
			t.Fatalf("Unexpected log output. expected log to contain: %v. Actual log: %v.", testString1, result)
		}
	case <-timeout:
		t.Fatal("Logger timed out!")
	}
}

func TestNewAmznLogger_LogLevel_ContainsGreaterLevelLogs(t *testing.T) {
	level := level.Error
	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName, LogLevel(level))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	timeout := time.After(defaultTimeout)
	logger.Fatal(testString1)
	select {
	case <-w.r:
		result := w.String()
		if !strings.Contains(result, testString1) {
			t.Fatalf("Unexpected log output. expected log to contain: %v. Actual log: %v.", testString1, result)
		}
	case <-timeout:
		t.Fatal("Logger timed out!")
	}
}

func TestNewAmznLogger_AddPackageLogLevel(t *testing.T) {
	defaultLevel := level.Fatal
	logger, err := NewAmznLogger(NewTestWriter(), defaultAppName, LogLevel(defaultLevel))
	if err != nil {
		t.Fatalf("Unexpected error returned. Error: %v", err)
	}

	var testPackageLevels = []struct {
		pack  string
		level level.Level
	}{
		{"go.amzn.com/foo/bar/baz", level.Error},
		{"go.amzn.com/foo/bar", level.Warn},
		{"go.amzn.com/foo", level.Info},
		{"go.amzn.com", level.Debug},
	}

	for _, tPL := range testPackageLevels {
		logger.AddPackageLogLevel(tPL.pack, tPL.level)
	}

	var testPackageLevelTests = []struct {
		pack  string
		level level.Level
	}{
		// Check each level
		{"go.amzn.com/foo/bar/baz.MyFunc", level.Error},
		{"go.amzn.com/foo/bar/baz.MyFunc.AnonymousFunc", level.Error},
		{"go.amzn.com/foo/bar.MyFunc", level.Warn},
		{"go.amzn.com/foo/bar.MyFunc.AnonymousFunc", level.Warn},
		{"go.amzn.com/foo.MyFunc", level.Info},
		{"go.amzn.com/foo.MyFunc.AnonymousFunc", level.Info},
		{"go.amzn.com.MyFunc", level.Debug},
		// Check deeper levels
		{"go.amzn.com/foo/bar/baz/aaa/bbb/ccc.MyFunc", level.Error},
		{"go.amzn.com/aaa/bbb/ccc.MyFunc", level.Debug},
		// Check multiple .s in the package name
		{"go.amzn.com/a.a.a/b.b.b/c.c.c.MyFunc", level.Debug},
		// Check no .s in the package name
		{"test.MyFunc", defaultLevel},
		// Test invalid funcs
		{"MyFunc", defaultLevel},
		{".MyFunc", defaultLevel},
		{"", defaultLevel},
	}

	// Default should stay default
	if logger.defaultLevel != defaultLevel {
		t.Fatalf("Expected log level: %v. Actual log level was: %v.", defaultLevel, logger.defaultLevel)
	}

	// Test the rest of the cases
	for _, tPLT := range testPackageLevelTests {
		if pkgLevel := logger.getLogLevel(tPLT.pack); pkgLevel != tPLT.level {
			t.Fatalf("Expected package %v log level: %v. Actual log level was: %v.", tPLT.pack, tPLT.level, pkgLevel)
		}
	}
}

// testLog uses genLogFn to call log functions, and match the output with expected values.
func testLog(t *testing.T, lvl level.Level, genLogFn func(logger *AmznLogger), expectedLogs []*regexp.Regexp) {
	t.Helper()

	w := NewTestWriter()
	logger, err := NewAmznLogger(w, defaultAppName, LogLevel(lvl))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	timeout := time.After(defaultTimeout)

	genLogFn(logger)

	var actualLogs string
	for i := 0; i < len(expectedLogs); i++ {
		select {
		case <-w.r:
			if i == len(expectedLogs)-1 {
				actualLogs = w.String()
			}
			continue
		case <-timeout:
			t.Fatalf("Unexpected timeout. Expected: %v.", testString1)
		}
	}

	logLines := strings.Split(actualLogs, "\n")
	for i := 0; i < len(expectedLogs); i++ {
		expectedLog := expectedLogs[i]
		actualLog := logLines[i]
		if !expectedLog.MatchString(actualLog) {
			t.Fatalf("Unexpected log. Expected: %v. Actual: %v.", expectedLog, actualLog)
		}
	}
}

func TestNewAmznLogger_Log(t *testing.T) {
	for i := 0; i < len(allLevels); i++ {
		lvl := allLevels[i]
		t.Run(level.LevelStr(lvl), func(t *testing.T) {
			testLog(t, lvl, logGenerator, logRegexes[i:])
		})
	}
}

func TestNewAmznLogger_Logf(t *testing.T) {
	for i := 0; i < len(allLevels); i++ {
		lvl := allLevels[i]
		t.Run(level.LevelStr(lvl), func(t *testing.T) {
			testLog(t, lvl, logfGenerator, logfRegexes[i:])
		})
	}
}

func TestNewAmznLogger_Logw(t *testing.T) {
	for i := 0; i < len(allLevels); i++ {
		lvl := allLevels[i]
		t.Run(level.LevelStr(lvl), func(t *testing.T) {
			testLog(t, lvl, logwGenerator, logwRegexes[i:])
		})
	}
}

func BenchmarkAmznLogger_GetPackageLogLevel(b *testing.B) {
	defaultLevel := level.Fatal
	logger, _ := NewAmznLogger(NewTestWriter(), defaultAppName, LogLevel(defaultLevel))

	b.ResetTimer()
	b.ReportAllocs()
	for i := 0; i < b.N; i++ {
		logger.getLogLevel("go.amzn.com/foo/bar/baz/aaa/bbb/ccc.MyFunc")
	}
}

func BenchmarkAmznLogger_GetPackageLogLevel_WithPackageLogLevels(b *testing.B) {
	defaultLevel := level.Fatal
	logger, _ := NewAmznLogger(NewTestWriter(), defaultAppName, LogLevel(defaultLevel))
	logger.AddPackageLogLevel("NeverMatch", defaultLevel)

	b.ResetTimer()
	b.ReportAllocs()
	for i := 0; i < b.N; i++ {
		logger.getLogLevel("go.amzn.com/foo/bar/baz/aaa/bbb/ccc.MyFunc")
	}
}

func TestNewAmznLogger_DiscardsLogs_WhenDiscardOnBufferFullFlagIsEnabled(t *testing.T) {
	timeout := 10 * time.Millisecond

	w := NewTestWriter()

	loggerBufferSize := 10
	logger, err := NewAmznLogger(w, defaultAppName, LogLevel(level.Info), BufferSize(loggerBufferSize), DiscardOnBufferFull(true))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}

	// Write more records than can be buffered by the writer and logger
	// so that the extra records get discarded by the logger.
	numberOfRecords := 2 * loggerBufferSize
	done := make(chan bool, 1)
	go func() {
		for counter := 0; counter < numberOfRecords; counter++ {
			logger.Info(counter)
		}
		done <- true
	}()

	select {
	case <-done:
	case <-time.After(timeout):
		t.Fatalf("Logger should not be blocked")
	}

	// If the logger's go routine which submits records to the writer is not fast
	// enough then atleast `loggerBufferSize` number of records will be written.
	expectedNumberOfRecordsMin := loggerBufferSize
	// If logger's go routine starts submitting the records to the writer before test is finished submitting records
	// to the logger then at max. `loggerBufferSize + 2` number of records will be written.
	// Why +2? a record is read in to a local variable from logger's buffer (writeChan) while it attempts to submit
	// that record to the writer which creates a space on the buffer. So, the logger can actually buffer `loggerBufferSize + 1`
	// records before getting blocked on the writer. And 1 record is buffered by the writer as well.
	expectedNumberOfRecordsMax := loggerBufferSize + 2

	actualNumberOfRecords := 0
	for i := 0; i < expectedNumberOfRecordsMax; i++ {
		select {
		case <-w.r:
			actualNumberOfRecords++
		case <-time.After(timeout):
			if actualNumberOfRecords < expectedNumberOfRecordsMin || actualNumberOfRecords > expectedNumberOfRecordsMax {
				t.Fatalf("Number of records should be between %d and %d. Actual: %d.", expectedNumberOfRecordsMin, expectedNumberOfRecordsMax, actualNumberOfRecords)
			}
		}
	}

	select {
	case <-w.r:
		t.Fatalf("Pending write not expected for any record.")
	case <-time.After(timeout):
	}

	verifyLogsAreWrittenInExpectedOrder(t, w.String())
}

func TestNewAmznLogger_DoesNotDiscardsLogs_WhenDiscardOnBufferFullIsDisabled(t *testing.T) {
	timeout := 10 * time.Millisecond

	w := NewTestWriter()

	loggerBufferSize := 10
	logger, err := NewAmznLogger(w, defaultAppName, LogLevel(level.Info), BufferSize(loggerBufferSize))
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}

	// Write more records than can be buffered by the writer and logger.
	numberOfRecords := 2 * loggerBufferSize
	// Write records in a separate go routine as this will block once
	// writer's and logger's buffers are full.
	done := make(chan bool, 1)
	go func() {
		for counter := 0; counter < numberOfRecords; counter++ {
			logger.Info(counter)
		}
		done <- true
	}()

	select {
	case <-done:
		t.Fatalf("Logger should be blocked")
	case <-time.After(timeout):
	}

	expectedNumberOfRecords := numberOfRecords

	actualNumberOfRecords := 0
	for i := 0; i < expectedNumberOfRecords; i++ {
		select {
		case <-w.r:
			actualNumberOfRecords++
		case <-time.After(timeout):
			t.Fatalf("Timed out waiting for %d records from the writer. Read %d records so far.", expectedNumberOfRecords, actualNumberOfRecords)
		}
	}

	select {
	case <-w.r:
		t.Fatalf("Pending write not expected for any record.")
	case <-time.After(timeout):
	}

	verifyLogsAreWrittenInExpectedOrder(t, w.String())
}

func verifyLogsAreWrittenInExpectedOrder(t *testing.T, logs string) {
	logLines := strings.Split(logs, "\n")
	previousCounter := -1
	for i := 0; i < len(logLines)-1; i++ {
		parts := strings.Split(logLines[i], " ")
		currentCounter, _ := strconv.Atoi(strings.TrimSpace(parts[len(parts)-1]))
		// Records should be written by the logger in the same order in which they were submitted
		// they might not be contiguous as it's not possible to determine which logs will be rejected by the logger.
		if currentCounter <= previousCounter {
			t.Fatalf("Unexpected log. Expected current counter to be larger than previous counter. PreviousCounter: %d, CurrentCounter: %d.", previousCounter, currentCounter)
		}
		previousCounter = currentCounter
	}
}

func TestNewAmznLogger_VerifyTimestampAreFromTimeOfEnqueue_UsingSlowWriter(t *testing.T) {
	numberOfLogs := 3
	waitGroup := &sync.WaitGroup{}
	waitGroup.Add(numberOfLogs)
	writerDelay := 10 * time.Millisecond
	writer := &slowWriter{
		delay:     writerDelay,
		logs:      []string{},
		waitGroup: waitGroup,
	}

	logger, _ := NewAmznLogger(writer,
		"test",
		DiscardOnBufferFull(false),
		BufferSize(1),
		DateFormat(time.RFC3339Nano),
		SetFormatOptions(FormatOptions{SkipAppName: true, SkipHostname: true, SkipFuncName: true, SkipPid: true}),
	)

	var logTimestamps []time.Time

	for i := 0; i < numberOfLogs; i++ {
		logTimestamps = append(logTimestamps, time.Now().UTC())
		logger.Info("this is line", i, "wrote @", time.Now().UTC().Format(time.RFC3339Nano))
	}

	waitGroup.Wait()

	if len(writer.logs) != numberOfLogs {
		t.Fatalf("Logs are missing from the writer. Expected: %d, Found: %d", numberOfLogs, len(writer.logs))
	}

	re := regexp.MustCompile(`(^.*)\s+\[INFO]`)
	for i, entry := range writer.logs {
		expectedTime := logTimestamps[i]
		parsedTime, err := time.Parse(time.RFC3339Nano, re.FindStringSubmatch(entry)[1])
		if err != nil {
			t.Fatalf("Failed to parse timestamp from entry %s", err)
		}
		diff := parsedTime.Sub(expectedTime).Nanoseconds()
		if diff > writerDelay.Nanoseconds()/2 || diff < 0 {
			t.Fatalf("The difference: [%s] between logged timestamp [%s] from the time log was submitted [%s] is more than the max allowed delay %s ", parsedTime.Sub(expectedTime), parsedTime, expectedTime, writerDelay/2)
		}
	}
}

type slowWriter struct {
	delay     time.Duration
	logs      []string
	waitGroup *sync.WaitGroup
}

func (w *slowWriter) Write(p []byte) (n int, err error) {
	defer w.waitGroup.Done()
	time.Sleep(w.delay)
	w.logs = append(w.logs, string(p))
	return fmt.Print("logged: ", string(p))
}
