/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package log_test

// This package exists to test the dynamic depth functionality of AmznLogger. A separate package is required
// as depth is determined using the package name. Moving the test to a separate package we can test that callstack depths
// are determined accurately.

import (
	"GoLog/log"
	"GoLog/log/level"
	"bytes"
	"strings"
	"testing"
	"time"
)

type TestWriter struct {
	buffer bytes.Buffer
	r      chan bool
}

func NewTestWriter() *TestWriter {
	writer := TestWriter{
		buffer: bytes.Buffer{},
		r:      make(chan bool, 1),
	}
	return &writer
}

func (t *TestWriter) Write(v []byte) (n int, err error) {
	n, err = t.buffer.Write(v)
	t.r <- true
	return n, err
}

func (t *TestWriter) String() string {
	return t.buffer.String()
}

func Test_Depth_PackFuncName(t *testing.T) {
	m := "test"
	w := NewTestWriter()
	packFuncName := "GoLog/log_test.Test_Depth_PackFuncName()"
	l, err := log.NewAmznLogger(w, "Tester")
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	timeout := time.After(1 * time.Second)
	l.Fatal(m)
	select {
	case <-w.r:
		result := w.String()
		if !strings.Contains(result, packFuncName) {
			t.Fatalf("Unexpected log output. Expected log to contain: %v. actual log: %v.", packFuncName, result)
		}
	case <-timeout:
		t.Fatal("Logger timed out!")
	}
}

type testLogWrapper struct {
	*log.AmznLogger
}

func (l *testLogWrapper) Info(format string, v ...interface{}) {
	l.Logf(1, level.Info, format, v...)
}

func Test_Depth_WrappedFuncName(t *testing.T) {
	m := "test"
	w := NewTestWriter()
	funcName := "GoLog/log_test.Test_Depth_WrappedFuncName()"
	l, err := log.NewAmznLogger(w, "Tester")
	if err != nil {
		t.Fatalf("Unexpected error returned. error: %v.", err)
	}
	wrapper := &testLogWrapper{l}
	wrapper.Info(m)
	timeout := time.After(1 * time.Second)
	select {
	case <-w.r:
		result := w.String()
		if !strings.Contains(result, funcName) {
			t.Fatalf("Unexpected log output. Expected log to contain: %v. actual log: %v.", funcName, result)
		}
	case <-timeout:
		t.Fatal("Logger timed out!")
	}
}
