package log

import (
	"fmt"
	"sync"
)

// Fields is the mapping of fields used for structured logging.
type Fields map[string]interface{}

// A FieldLogger represents an active logging object that contains fields to be
// included in all outputs.
type FieldLogger struct {
	fields    Fields
	fieldLock sync.RWMutex
}

// NewFieldLogger instantiates a new FieldLogger with an empty field mapping.
func NewFieldLogger() *FieldLogger {
	return &FieldLogger{fields: Fields{}}
}

// Returns a new field mapping that is merged from multiple mappings.
func (logger *FieldLogger) mergeFieldMappings(fieldMaps ...Fields) Fields {
	logger.fieldLock.RLock()
	defer logger.fieldLock.RUnlock()
	merged := make(Fields)
	for k, v := range logger.fields {
		merged[k] = v
	}
	for _, fields := range fieldMaps {
		for k, v := range fields {
			merged[k] = v
		}
	}
	return merged
}

// WithField sets a new field to be tracked and printed in all future logs.
func (logger *FieldLogger) WithField(k string, v interface{}) {
	logger.fieldLock.Lock()
	defer logger.fieldLock.Unlock()
	logger.fields[k] = v
}

// WithFields sets multiple new fields to be tracked and printed in all future logs.
func (logger *FieldLogger) WithFields(fields Fields) {
	logger.fieldLock.Lock()
	defer logger.fieldLock.Unlock()
	for k, v := range fields {
		logger.fields[k] = v
	}
}

func (logger *FieldLogger) Trace(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Tracew(logger.mergeFieldMappings(), v...)
	}
}

func (logger *FieldLogger) Tracef(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Tracew(logger.mergeFieldMappings(), fmt.Sprintf(format, v...))
	}
}

func (logger *FieldLogger) Tracew(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Tracew(logger.mergeFieldMappings(fields), v...)
	}
}

func (logger *FieldLogger) Debug(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Debugw(logger.mergeFieldMappings(), v...)
	}
}

func (logger *FieldLogger) Debugf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Debugw(logger.mergeFieldMappings(), fmt.Sprintf(format, v...))
	}
}

func (logger *FieldLogger) Debugw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Debugw(logger.mergeFieldMappings(fields), v...)
	}
}

func (logger *FieldLogger) Info(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Infow(logger.mergeFieldMappings(), v...)
	}
}

func (logger *FieldLogger) Infof(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Infow(logger.mergeFieldMappings(), fmt.Sprintf(format, v...))
	}
}

func (logger *FieldLogger) Infow(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Infow(logger.mergeFieldMappings(fields), v...)
	}
}

func (logger *FieldLogger) Warn(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Warnw(logger.mergeFieldMappings(), v...)
	}
}

func (logger *FieldLogger) Warnf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Warnw(logger.mergeFieldMappings(), fmt.Sprintf(format, v...))
	}
}

func (logger *FieldLogger) Warnw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Warnw(logger.mergeFieldMappings(fields), v...)
	}
}

func (logger *FieldLogger) Error(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Errorw(logger.mergeFieldMappings(), v...)
	}
}

func (logger *FieldLogger) Errorf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Errorw(logger.mergeFieldMappings(), fmt.Sprintf(format, v...))
	}
}

func (logger *FieldLogger) Errorw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Errorw(logger.mergeFieldMappings(fields), v...)
	}
}

func (logger *FieldLogger) Fatal(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Fatalw(logger.mergeFieldMappings(), v...)
	}
}

func (logger *FieldLogger) Fatalf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Fatalw(logger.mergeFieldMappings(), fmt.Sprintf(format, v...))
	}
}

func (logger *FieldLogger) Fatalw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Fatalw(logger.mergeFieldMappings(fields), v...)
	}
}
