package log

import (
	"fmt"
	"reflect"
	"strconv"
	"sync"
	"testing"
)

func TestFieldLogger_mergeFieldMappings(t *testing.T) {
	field1 := Fields{"1": "a", "2": "b", "3": "c"}
	field2 := Fields{"2": "d", "3": "e", "4": "f"}
	field3 := Fields{"3": "g", "4": "h", "5": "i"}
	expected := Fields{"One": "1", "1": "a", "2": "d", "3": "g", "4": "h", "5": "i"}
	logger := NewFieldLogger()
	logger.WithField("One", "1")

	merged := logger.mergeFieldMappings(field1, field2, field3)

	if len(expected) != len(merged) {
		t.Fatal("Unexpected merge result.")
	}
	for k, v := range expected {
		if v != merged[k] {
			t.Fatal("Unexpected merge result.")
		}
	}
}

func TestFieldLogger_WithField(t *testing.T) {
	logger := NewFieldLogger()
	logger.WithField("key", "value")
	if len(logger.fields) != 1 || logger.fields["key"] != "value" {
		t.Fatal("Unexpected fields from WithField() call.")
	}
}

func TestFieldLogger_WithFields(t *testing.T) {
	logger := NewFieldLogger()
	logger.WithFields(Fields{"1": "a", "2": "b"})
	if len(logger.fields) != 2 || logger.fields["1"] != "a" || logger.fields["2"] != "b" {
		t.Fatal("Unexpected fields from WithFields() call.")
	}
}

func TestFieldLogger_Trace(t *testing.T) {
	defer SetLogger(nil)
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockTracew: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Trace(expectedLog...)
}

func TestFieldLogger_Tracef(t *testing.T) {
	defer SetLogger(nil)
	expectedFormat := "%s %d %.1f"
	expectedLogComponents := []interface{}{"a", 1, 1.5}
	expectedLog := []interface{}{fmt.Sprintf(expectedFormat, expectedLogComponents...)}

	SetLogger(&MockAmznLogger{
		MockTracew: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Tracef(expectedFormat, expectedLogComponents...)
}

func TestFieldLogger_Tracew(t *testing.T) {
	defer SetLogger(nil)
	expectedFields := Fields{"k": "v"}
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockTracew: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedFields, fields) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Tracew(expectedFields, expectedLog...)
}

func TestFieldLogger_Debug(t *testing.T) {
	defer SetLogger(nil)
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockDebugw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Debug(expectedLog...)
}

func TestFieldLogger_Debugf(t *testing.T) {
	defer SetLogger(nil)
	expectedFormat := "%s %d %.1f"
	expectedLogComponents := []interface{}{"a", 1, 1.5}
	expectedLog := []interface{}{fmt.Sprintf(expectedFormat, expectedLogComponents...)}

	SetLogger(&MockAmznLogger{
		MockDebugw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Debugf(expectedFormat, expectedLogComponents...)
}

func TestFieldLogger_Debugw(t *testing.T) {
	defer SetLogger(nil)
	expectedFields := Fields{"k": "v"}
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockDebugw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedFields, fields) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Debugw(expectedFields, expectedLog...)
}

func TestFieldLogger_Info(t *testing.T) {
	defer SetLogger(nil)
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockInfow: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Info(expectedLog...)
}

func TestFieldLogger_Infof(t *testing.T) {
	defer SetLogger(nil)
	expectedFormat := "%s %d %.1f"
	expectedLogComponents := []interface{}{"a", 1, 1.5}
	expectedLog := []interface{}{fmt.Sprintf(expectedFormat, expectedLogComponents...)}

	SetLogger(&MockAmznLogger{
		MockInfow: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Infof(expectedFormat, expectedLogComponents...)
}

func TestFieldLogger_Infow(t *testing.T) {
	defer SetLogger(nil)
	expectedFields := Fields{"k": "v"}
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockInfow: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedFields, fields) {
				t.Errorf("Expected: %v. Actual: %v.", expectedFields, fields)
			}
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Infow(expectedFields, expectedLog...)
}

func TestFieldLogger_Warn(t *testing.T) {
	defer SetLogger(nil)
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockWarnw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Warn(expectedLog...)
}

func TestFieldLogger_Warnf(t *testing.T) {
	defer SetLogger(nil)
	expectedFormat := "%s %d %.1f"
	expectedLogComponents := []interface{}{"a", 1, 1.5}
	expectedLog := []interface{}{fmt.Sprintf(expectedFormat, expectedLogComponents...)}

	SetLogger(&MockAmznLogger{
		MockWarnw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Warnf(expectedFormat, expectedLogComponents...)
}

func TestFieldLogger_Warnw(t *testing.T) {
	defer SetLogger(nil)
	expectedFields := Fields{"k": "v"}
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockWarnw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedFields, fields) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Warnw(expectedFields, expectedLog...)
}

func TestFieldLogger_Error(t *testing.T) {
	defer SetLogger(nil)
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockErrorw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Error(expectedLog...)
}

func TestFieldLogger_Errorf(t *testing.T) {
	defer SetLogger(nil)
	expectedFormat := "%s %d %.1f"
	expectedLogComponents := []interface{}{"a", 1, 1.5}
	expectedLog := []interface{}{fmt.Sprintf(expectedFormat, expectedLogComponents...)}

	SetLogger(&MockAmznLogger{
		MockErrorw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Errorf(expectedFormat, expectedLogComponents...)
}

func TestFieldLogger_Errorw(t *testing.T) {
	defer SetLogger(nil)
	expectedFields := Fields{"k": "v"}
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockErrorw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedFields, fields) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Errorw(expectedFields, expectedLog...)
}

func TestFieldLogger_Fatal(t *testing.T) {
	defer SetLogger(nil)
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockFatalw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Fatal(expectedLog...)
}

func TestFieldLogger_Fatalf(t *testing.T) {
	defer SetLogger(nil)
	expectedFormat := "%s %d %.1f"
	expectedLogComponents := []interface{}{"a", 1, 1.5}
	expectedLog := []interface{}{fmt.Sprintf(expectedFormat, expectedLogComponents...)}

	SetLogger(&MockAmznLogger{
		MockFatalw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Log mismatch. Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Fatalf(expectedFormat, expectedLogComponents...)
}

func TestFieldLogger_Fatalw(t *testing.T) {
	defer SetLogger(nil)
	expectedFields := Fields{"k": "v"}
	expectedLog := []interface{}{"a", 1, 1.5}

	SetLogger(&MockAmznLogger{
		MockFatalw: func(fields Fields, v ...interface{}) {
			if !reflect.DeepEqual(expectedFields, fields) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
			if !reflect.DeepEqual(expectedLog, v) {
				t.Errorf("Expected: %v. Actual: %v.", expectedLog, v)
			}
		},
	})

	logger := NewFieldLogger()
	logger.Fatalw(expectedFields, expectedLog...)
}

func TestFieldLogger_ConcurrentReadWriteField(t *testing.T) {
	logger := NewFieldLogger()
	logger.WithField("test", "abc")

	methods := []func(fields Fields, v ...interface{}){logger.Tracew, logger.Debugw, logger.Infow, logger.Warnw, logger.Errorw}

	for _, loggerMethod := range methods {
		runFieldLoggerConcurrentReadWriteFieldTest(logger.WithFields, loggerMethod)
	}
}

func TestFieldLogger_ConcurrentReadWriteField_NoFieldArgs(t *testing.T) {
	logger := NewFieldLogger()
	logger.WithField("test-1", "abc")

	methods := []func(v ...interface{}){logger.Trace, logger.Debug, logger.Info, logger.Warn, logger.Error}

	for _, loggerMethod := range methods {
		runFieldLoggerConcurrentReadWriteFieldTest(logger.WithFields, func(fields Fields, v ...interface{}) {
			loggerMethod(v...)
		})
	}
}

func TestFieldLogger_ConcurrentReadWriteField_WithField(t *testing.T) {
	logger := NewFieldLogger()
	logger.WithFields(Fields{"one": "1", "two": "2"})

	methods := []func(fields Fields, v ...interface{}){logger.Tracew, logger.Debugw, logger.Infow, logger.Warnw, logger.Errorw}

	for _, loggerMethod := range methods {
		runFieldLoggerConcurrentReadWriteFieldTest(func(fields Fields) {
			for k, v := range fields {
				logger.WithField(k, v)
			}
		}, loggerMethod)
	}
}

func TestFieldLogger_ConcurrentReadWriteField_NoFieldArgs_WithField(t *testing.T) {
	logger := NewFieldLogger()
	logger.WithFields(Fields{"one": "1", "two": "2"})

	methods := []func(v ...interface{}){logger.Trace, logger.Debug, logger.Info, logger.Warn, logger.Error}

	for _, loggerMethod := range methods {
		runFieldLoggerConcurrentReadWriteFieldTest(func(fields Fields) {
			for k, v := range fields {
				logger.WithField(k, v)
			}
		}, func(fields Fields, v ...interface{}) {
			loggerMethod(v...)
		})
	}
}

func runFieldLoggerConcurrentReadWriteFieldTest(withField func(fields Fields), loggerMethod func(fields Fields, v ...interface{})) {
	var wg sync.WaitGroup
	n := 10
	wg.Add(n)
	for i := 0; i < n; i++ {
		go func(iter int) {
			defer wg.Done()
			if iter%2 == 0 {
				withField(Fields{"test": strconv.Itoa(iter)})
			} else {
				loggerMethod(map[string]interface{}{"test": strconv.Itoa(iter)})
			}

		}(i)
	}
	wg.Wait()
}
