/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package level

import (
	"strings"
)

type Level int

const (
	Off Level = 1 + iota
	Trace
	Debug
	Info
	Warn
	Error
	Fatal
)

var strToLevel = map[string]Level{
	"off": Off, "trace": Trace, "debug": Debug, "info": Info, "warn": Warn, "error": Error, "fatal": Fatal,
}

// LevelStr turns the given Level into a bracketed string version of the level.
func LevelStr(level Level) string {
	switch level {
	case Off:
		return ""
	case Trace:
		return "[TRACE]"
	case Debug:
		return "[DEBUG]"
	case Info:
		return "[INFO]"
	case Warn:
		return "[WARN]"
	case Error:
		return "[ERROR]"
	case Fatal:
		return "[FATAL]"
	}
	return "[UNKNOWN]"
}

// FromString translates the given string into a Level if it matches a known
// level.  Otherwise returns Off.
func FromString(level string) Level {
	level = strings.ToLower(level)
	if lvl, ok := strToLevel[level]; ok {
		return lvl
	}
	return Off
}
