/* Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package log

import (
	"io"
)

var globalLogger Logger

type Logger interface {
	Trace(v ...interface{})
	Tracef(format string, v ...interface{})
	Tracew(fields Fields, v ...interface{})
	Debug(v ...interface{})
	Debugf(format string, v ...interface{})
	Debugw(fields Fields, v ...interface{})
	Info(v ...interface{})
	Infof(format string, v ...interface{})
	Infow(fields Fields, v ...interface{})
	Warn(v ...interface{})
	Warnf(format string, v ...interface{})
	Warnw(fields Fields, v ...interface{})
	Error(v ...interface{})
	Errorf(format string, v ...interface{})
	Errorw(fields Fields, v ...interface{})
	Fatal(v ...interface{})
	Fatalf(format string, v ...interface{})
	Fatalw(fields Fields, v ...interface{})
}

// SetLogger sets the global logger to the specified logger instance.
func SetLogger(logger Logger) {
	globalLogger = logger
}

// Close closes the global logger if possible (it must implement the io.Closer interface) then removes it.
func Close() error {
	if globalLogger == nil {
		return nil
	} else if s, ok := globalLogger.(io.Closer); ok {
		globalLogger = nil
		return s.Close()
	} else {
		globalLogger = nil
		return nil
	}
}

// Trace prints the Trace log with the global logger.
func Trace(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Trace(v...)
	}
}

// Tracef prints the Trace log with the global logger.
func Tracef(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Tracef(format, v...)
	}
}

// Tracew prints the Trace log with extra fields with the global logger.
func Tracew(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Tracew(fields, v...)
	}
}

// Debug prints the Debug log with the global logger.
func Debug(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Debug(v...)
	}
}

// Debugf prints the Debug log with the global logger.
func Debugf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Debugf(format, v...)
	}
}

// Debugw prints the Debug log with extra fields with the global logger.
func Debugw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Debugw(fields, v...)
	}
}

// Info prints the Info log with the global logger.
func Info(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Info(v...)
	}
}

// Infof prints the Info log with the global logger.
func Infof(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Infof(format, v...)
	}
}

// Infow prints the Info log with the global logger.
func Infow(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Infow(fields, v...)
	}
}

// Warn prints the Warn log with the global logger.
func Warn(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Warn(v...)
	}
}

// Warnf prints the Warn log with the global logger.
func Warnf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Warnf(format, v...)
	}
}

// Warnw prints the Warn log with extra fields with the global logger.
func Warnw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Warnw(fields, v...)
	}
}

// Error prints the Error log with the global logger.
func Error(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Error(v...)
	}
}

// Errorf prints the Error log with the global logger.
func Errorf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Errorf(format, v...)
	}
}

// Errorw prints the Error log with extra fields with the global logger.
func Errorw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Errorw(fields, v...)
	}
}

// Fatal prints the Fatal log with the global logger.
func Fatal(v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Fatal(v...)
	}
}

// Fatalf prints the Fatal log with the global logger.
func Fatalf(format string, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Fatalf(format, v...)
	}
}

// Fatalw prints the Fatal log with extra fields with the global logger.
func Fatalw(fields Fields, v ...interface{}) {
	if globalLogger != nil {
		globalLogger.Fatalw(fields, v...)
	}
}
