package odin

import (
	"encoding/base64"
	"encoding/json"
	"errors"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

const (
	Timeout = time.Second * 5 // Timeout after this
)

var (
	client = http.Client{Timeout: Timeout}

	// Endpoint is used to hit the odin daemon.
	// custom endpoints for mocking out responses for testing.
	// For backwards compatibility, left as exported
	Endpoint = DefaultOdinEndpoint
)

/*
{"Error":{"Type":"Sender","Message":"Cannot find material 'com.amazon.prediction.dao.OdinMWSCredential.??'. (Reason: Serial ?? not found for materials ('Principal') named 'com.amazon.prediction.dao.OdinMWSCredential')","Code":"MaterialNotFoundException"}}
*/

type retrieveResult struct {
	Material *RawMaterial `json:"material"`
	Error    *retrieveError
}

type retrieveError struct {
	Type    string
	Message string
}

func (err retrieveError) Error() string {
	return err.Message
}

type odinDaemon struct {
}

func (o *odinDaemon) Retrieve(materialSet, materialType string, serial *int64) (*RawMaterial, error) {

	query := url.Values{}
	query.Set("material.materialName", materialSet)
	query.Set("material.materialType", materialType)
	query.Set("ContentType", "JSON")
	query.Set("Operation", "Retrieve")
	if serial != nil {
		query.Set("material.materialSerial", strconv.FormatInt(*serial, 10))
	}

	request, _ := http.NewRequest("GET", Endpoint, nil)
	request.URL.RawQuery = query.Encode()

	resp, err := client.Do(request)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()

	var result retrieveResult
	b, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal(b, &result)
	if err != nil {
		return nil, err
	}

	// The odin local daemon doesn't return an 'Active' field like OdinEverywhere,
	// so set the boolean to true
	if result.Material != nil {
		result.Material.Active = true
	}

	switch {
	case result.Error != nil:
		return result.Material, result.Error
	case result.Material == nil:
		return nil, errors.New("Couldn't parse response")
	default:
		data, err := base64.StdEncoding.DecodeString(result.Material.Base64MaterialData)
		if err == nil {
			result.Material.MaterialData = data
		}
		return result.Material, err
	}
}
