package odin

import (
	"crypto/sha256"
	"encoding/xml"
	"fmt"
	"io/ioutil"
	"path/filepath"
)

var (
	// OdinEverywhere path on the filesystem
	odinEverywherePath = DefaultOdinEverywherePath
)

type materialList struct {
	Materials []RawMaterial `xml:"Material"`
}

// Extra functions to support OdinEverywhere file loading
func (x *RawMaterial) hashKey() string {
	return materialSetHashKey(x.MaterialName + "." + fmt.Sprintf("%d", x.MaterialSerial) + "." + x.MaterialType)
}

func (x *RawMaterial) loadMaterialData() error {
	fn := filepath.Join(odinEverywherePath, x.hashKey())

	data, err := ioutil.ReadFile(fn)
	if err != nil {
		return err
	}

	x.MaterialData = data

	return nil
}

func materialSetHashKey(mset string) string {
	sum := sha256.Sum256([]byte(mset))
	return fmt.Sprintf("%x", sum)
}

type odinEverywhere struct {
}

func (o *odinEverywhere) Retrieve(materialSet, materialType string, serial *int64) (*RawMaterial, error) {
	manifestFn := filepath.Join(odinEverywherePath, materialSetHashKey(materialSet)+".xml")
	xmlData, err := ioutil.ReadFile(manifestFn)
	if err != nil {
		return nil, err
	}

	xmlMaterials := materialList{}
	err = xml.Unmarshal(xmlData, &xmlMaterials)
	if err != nil {
		return nil, err
	}

	for _, material := range xmlMaterials.Materials {
		material.MaterialName = materialSet
		// The Java OdinLocalRetriver for OdinEverywhere filters results by IsActive() before returning them
		if material.MaterialType == materialType && material.IsActive() {
			// XML is ordered by descending serials
			if serial == nil {
				material.loadMaterialData()
				return &material, nil
			} else if *serial == material.MaterialSerial {
				material.loadMaterialData()
				return &material, nil
			}
		}
	}

	if serial != nil {
		return nil, fmt.Errorf("Material %s / Serial %d could not be found", materialSet, *serial)
	} else {
		return nil, fmt.Errorf("Material %s could not be found", materialSet)
	}
}
