package odin

import (
	"bytes"
	"encoding/base64"
	"path/filepath"
	"runtime"
	"testing"
)

const (
	CredentialPairMaterialSet           = "com.amazon.testapp.credentials"
	CredentialPairMaterialSetLongSerial = "com.amazon.testapp.credentials.long.serial"
	PrivateKeyMaterialSet               = "com.amazon.testapp.key"
	CertificateMaterialSet              = "com.amazon.testapp.certificates"
	SymmetricKeyMaterialSet             = "com.amazon.testapp.symmetric-key"

	PrincipalMaterialData    = "ABCDEFGHIJKLMNOP\n"
	CredentialMaterialData   = "PONMLKJIHGFEDCBA\n"
	PrivateKeyMaterialData   = "QWERTYUIOPASDFGHJKLZXCVBNM\n"
	SymmetricKeyMaterialData = "lpfHRmNwQVkWIcdhpVg12m4pGvR66R7ssiVu2F9b+v0="
)

func init() {
	_, fn, _, _ := runtime.Caller(0)

	// Load from testfiles
	SwitchToOdinEverywhere(filepath.Join(filepath.Dir(fn), "testFiles"))
}

func TestSerialNotGiven(t *testing.T) {
	mat, err := odinEverywhereImplementation.Retrieve(CredentialPairMaterialSet, "Principal", nil)
	if err != nil {
		t.Fatal(err)
	}

	if bytes.Compare([]byte(PrincipalMaterialData), mat.MaterialData) != 0 {
		t.Fatalf("materialData is %+v but expected %+v", mat.MaterialData, []byte(PrincipalMaterialData))
	}

	if mat.NotBefore.Unix() != 1471284860 {
		t.Fatalf("NotBefore is %d but expected 1471284860", mat.NotBefore.Unix())
	}

	if mat.NotAfter.Unix() != 31471284860 {
		t.Fatalf("NotAfter is %d but expected 31471284860", mat.NotAfter.Unix())
	}
}

func TestMaterialNotFoundInMaterialSet(t *testing.T) {
	// Material set should be empty
	_, err := odinEverywhereImplementation.Retrieve(CertificateMaterialSet, "Certificate", nil)
	if err == nil {
		t.Fatalf("Expected error because material set is empty")
	}
}

func TestGetMaterialWithSpecificSerial(t *testing.T) {
	serial := int64(1)
	mat, err := odinEverywhereImplementation.Retrieve(PrivateKeyMaterialSet, "PrivateKey", &serial)
	if err != nil {
		t.Fatal(err)
	}

	if mat.MaterialSerial != 1 {
		t.Fatalf("Serial is %d but expected 1", mat.MaterialSerial)
	}

	if bytes.Compare([]byte(PrivateKeyMaterialData), mat.MaterialData) != 0 {
		t.Fatalf("materialData is %+v but expected %+v", mat.MaterialData, []byte(PrivateKeyMaterialData))
	}
}

func TestGetMaterialWithLatestSerial(t *testing.T) {
	mat, err := odinEverywhereImplementation.Retrieve(PrivateKeyMaterialSet, "PrivateKey", nil)
	if err != nil {
		t.Fatal(err)
	}

	if mat.MaterialSerial != 2 {
		t.Fatalf("Serial is %d but expected 2", mat.MaterialSerial)
	}

	if bytes.Compare([]byte(PrivateKeyMaterialData), mat.MaterialData) != 0 {
		t.Fatalf("materialData is %+v but expected %+v", mat.MaterialData, []byte(PrivateKeyMaterialData))
	}
}

func TestGetMaterialPair(t *testing.T) {
	pMat, err := odinEverywhereImplementation.Retrieve(CredentialPairMaterialSet, "Principal", nil)
	if err != nil {
		t.Fatal(err)
	}

	cMat, err := odinEverywhereImplementation.Retrieve(CredentialPairMaterialSet, "Credential", nil)
	if err != nil {
		t.Fatal(err)
	}

	if pMat.MaterialSerial != 2 {
		t.Fatalf("principal serial is %d but expected 2", pMat.MaterialSerial)
	}

	if cMat.MaterialSerial != 1264770641012 {
		t.Fatalf("principal serial is %d but expected 1264770641012", cMat.MaterialSerial)
	}

	if bytes.Compare([]byte(PrincipalMaterialData), pMat.MaterialData) != 0 {
		t.Fatalf("principal materialData is %+v but expected %+v",
			pMat.MaterialData, []byte(PrincipalMaterialData))
	}

	if bytes.Compare([]byte(CredentialMaterialData), cMat.MaterialData) != 0 {
		t.Fatalf("credential materialData is %+v but expected %+v",
			cMat.MaterialData, []byte(CredentialMaterialData))
	}
}

func TestGetMaterialPairWithSpecificSerial(t *testing.T) {
	serial := int64(1)
	pMat, err := odinEverywhereImplementation.Retrieve(CredentialPairMaterialSet, "Principal", &serial)
	if err != nil {
		t.Fatal(err)
	}

	cMat, err := odinEverywhereImplementation.Retrieve(CredentialPairMaterialSet, "Credential", &serial)
	if err != nil {
		t.Fatal(err)
	}

	if bytes.Compare([]byte(PrincipalMaterialData), pMat.MaterialData) != 0 {
		t.Fatalf("principal materialData is %+v but expected %+v",
			pMat.MaterialData, []byte(PrincipalMaterialData))
	}

	if bytes.Compare([]byte(CredentialMaterialData), cMat.MaterialData) != 0 {
		t.Fatalf("credential materialData is %+v but expected %+v",
			cMat.MaterialData, []byte(CredentialMaterialData))
	}
}

func TestGetMaterialPairWithLongSerial(t *testing.T) {
	serial := int64(1264770641012)
	pMat, err := odinEverywhereImplementation.Retrieve(CredentialPairMaterialSetLongSerial, "Principal", &serial)
	if err != nil {
		t.Fatal(err)
	}

	cMat, err := odinEverywhereImplementation.Retrieve(CredentialPairMaterialSetLongSerial, "Credential", &serial)
	if err != nil {
		t.Fatal(err)
	}

	if pMat.MaterialSerial != 1264770641012 {
		t.Fatalf("principal material has serial %d but expected 1264770641012", pMat.MaterialSerial)
	}

	if cMat.MaterialSerial != 1264770641012 {
		t.Fatalf("credential material has serial %d but expected 1264770641012", cMat.MaterialSerial)
	}

	if bytes.Compare([]byte(PrincipalMaterialData), pMat.MaterialData) != 0 {
		t.Fatalf("principal materialData is %+v but expected %+v",
			pMat.MaterialData, []byte(PrincipalMaterialData))
	}

	if bytes.Compare([]byte(CredentialMaterialData), cMat.MaterialData) != 0 {
		t.Fatalf("credential materialData is %+v but expected %+v",
			cMat.MaterialData, []byte(CredentialMaterialData))
	}
}

func TestGetSymmetricKey(t *testing.T) {
	mat, err := odinEverywhereImplementation.Retrieve(SymmetricKeyMaterialSet, "SymmetricKey", nil)
	if err != nil {
		t.Fatal(err)
	}

	if base64.StdEncoding.EncodeToString(mat.MaterialData) != SymmetricKeyMaterialData {
		t.Fatalf("Symmetric Key data is %s but expected %s",
			base64.StdEncoding.EncodeToString(mat.MaterialData), SymmetricKeyMaterialData)
	}
}
