package odin_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"odin"
)

func TestErrorCase(t *testing.T) {
	ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		mockData := `{"Error":{"Type":"Sender","Message":"Cannot find material 'com.amazon.prediction.dao.OdinMWSCredential.??'. (Reason: Serial ?? not found for materials ('Principal') named 'com.amazon.prediction.dao.OdinMWSCredential')","Code":"MaterialNotFoundException"}}`
		fmt.Fprintln(w, mockData)
	}))

	defer ts.Close()

	odin.SwitchToDaemon(ts.URL)
	_, _, err := odin.CredentialPair("foo.bar.qux")
	if err == nil {
		t.Errorf("Expected error, but received nil")
	}
}

func TestValidCase(t *testing.T) {
	errJSON := `{"Error":{"Type":"Sender","Message":"Cannot find material 'com.amazon.prediction.dao.OdinMWSCredential.??'. (Reason: Serial ?? not found for materials ('Principal') named 'com.amazon.prediction.dao.OdinMWSCredential')","Code":"MaterialNotFoundException"}}`
	ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		types := map[string]string{
			"Credential": `{"material":{"materialType":"Credential","notAfter":32939560450,"materialData":"Y3JlZGVudGlhbA==","materialSerial":5,"notBefore":1403560450,"materialName":"foo.bar.qux"}}`,
			"Principal":  `{"material":{"materialType":"Principal","notAfter":32939560450,"materialData":"cHJpbmNpcGFs","materialSerial":5,"notBefore":1403560450,"materialName":"foo.bar.qux"}}`,
		}

		t := r.URL.Query().Get("material.materialType")
		v, ok := types[t]
		if !ok {
			fmt.Fprintln(w, errJSON)
		} else {
			fmt.Fprintln(w, v)
		}
	}))

	defer ts.Close()

	odin.SwitchToDaemon(ts.URL)
	_, _, err := odin.CredentialPair("foo.bar.qux")
	if err != nil {
		t.Errorf("Expected nil, but received %v", err)
	}
}

func TestRawRetrieve(t *testing.T) {
	fake := &fakeImpl{
		err: fmt.Errorf("some error"),
	}
	odin.SwitchToCustom(fake)
	if mat, err := odin.Retrieve("some-material", "some-type", odin.Serial(42)); mat != &fake.material {
		t.Error("odin.Retrieve did not return the material produced by fakeImpl")
	} else if err != fake.err {
		t.Error("odin.Retrieve did not return the error produced by fakeImpl")
	}
	if want, got := "m: some-material, t: some-type, s: 42", fake.sideEffect; want != got {
		t.Log("want: ", want)
		t.Error("got: ", got)
	}
}

type fakeImpl struct {
	material   odin.RawMaterial
	err        error
	sideEffect string
}

func (fake *fakeImpl) Retrieve(materialSet, materialType string, serial *int64) (*odin.RawMaterial, error) {
	fake.sideEffect = fmt.Sprintf("m: %s, t: %s, s: %v", materialSet, materialType, *serial)
	return &fake.material, fake.err
}
