#!/usr/bin/python -tt
import json
import random
import string

from re import search

from locust import HttpLocust, TaskSet, task

# Uses the locust test/load framework
# it will look for locustfile.py in current workdir
#
# run like: locust --host=http://<fqdn>:<port> --no-web -c <usercount> -r <hatchrate> -t <time>

# pattern to guard against deregistering the cluster members, should be re.search()-able
# in my test env consul nodes are just called consul1, consul2, consul3 so the pattern is simple
PATTERN = r'consul'


# generate synthetic data
# Changing the arguments used when calling this will change the entropy so that can be used as a way of
# for example increasing the chance that you hit an existing KV item with get_kv_item or post_kv_item.
def data_generator(length=8, chars=string.ascii_uppercase + string.digits):
    return ''.join(random.choice(chars) for _ in range(length))


# generate synthetic ipv4 address
# XXX This may need to only generate within a given CIDR, current addresses can be a bit, wild.
def ipv4_generator():
    return '.'.join(str(random.randint(0, 255)) for _ in range(4))


# The tests below are Locust tasks and they all invoke some form of action towards the consul HTTP API
# example payloads here: https://www.consul.io/api/index.html
#
# In the @task(n) statement, the n denotes the weight of that test.
# If you want specific tests to run more often, increase their relative weight.
class TestConsulHTTP(TaskSet):
    @task(1)
    def put_kv_item(self):
        # Puts an item into the KV store
        print("instance {} putting a KV item".format(self.locust))
        key_name = data_generator(4, "12345678")
        key_value = data_generator()
        r = self.client.put("/v1/kv/{}".format(key_name), "{}".format(key_value), name="put /kv/*")
        return

    @task(5)
    def get_kv_item(self):
        # Attempts to read an item from the KV store, item may not exist so this usually 404:s
        key_name = data_generator(4, "12345678")
        print("instance {} attempting to get a KV item {}".format(self.locust, key_name))
        r = self.client.get("/v1/kv/{}?stale".format(key_name), name="get /kv/*")
        return

    @task(1)
    def get_metrics(self):
        # Fetches the metrics from the cluster
        print("instance {} reading metrics".format(self.locust))
        r = self.client.get("/v1/agent/metrics", name="get /agent/metrics")
        return

    @task(5)
    def get_catalog_nodes(self):
        # Fetch the list of datacenters and then fetch the list of nodes in a random datacenter
        print("instance {} fetching catalog".format(self.locust))
        r = self.client.get("/v1/catalog/datacenters", name="get /catalog/datacenters")
        DATACENTER_LIST = r.json()
        dc = random.choice(DATACENTER_LIST)
        print("instance {} fetching nodes for {}".format(self.locust, dc))
        r = self.client.get("/v1/catalog/nodes?dc={}&stale".format(dc), name="get /catalog/nodes?dc=*")
        return

    @task(3)
    def get_blocking_stale_services(self):
        # Fetch the list of datacenters and then fetch the list of services in a random datacenter
        print("instance {} fetching catalog".format(self.locust))
        r = self.client.get("/v1/catalog/datacenters", name="get /catalog/datacenters")
        DATACENTER_LIST = r.json()
        dc = random.choice(DATACENTER_LIST)
        print("instance {} fetching stale blocking services for {}".format(self.locust, dc))
        r = self.client.get("/v1/health/service/consul?dc={}&passing=1&stale&index=9999999999".format(dc), name="get /catalog/services?dc=*")
        return

    @task(3)
    def get_blocking_services(self):
        # Fetch the list of datacenters and then fetch the list of services in a random datacenter
        print("instance {} fetching catalog".format(self.locust))
        r = self.client.get("/v1/catalog/datacenters", name="get /catalog/datacenters")
        DATACENTER_LIST = r.json()
        dc = random.choice(DATACENTER_LIST)
        print("instance {} fetching non-stale blocking services for {}".format(self.locust, dc))
        r = self.client.get("/v1/health/service/consul?dc={}&passing=1&index=9999999999".format(dc), name="get /catalog/services?dc=*")
        return

    @task(3)
    def get_services(self):
        # Fetch the list of datacenters and then fetch the list of services in a random datacenter
        print("instance {} fetching catalog".format(self.locust))
        r = self.client.get("/v1/catalog/datacenters", name="get /catalog/datacenters")
        DATACENTER_LIST = r.json()
        dc = random.choice(DATACENTER_LIST)
        print("instance {} fetching services for {}".format(self.locust, dc))
        r = self.client.get("/v1/catalog/services?dc={}&stale".format(dc), name="get /catalog/services?dc=*")
        return

    @task(0)
    def register_node(self):
        # Fetch the list of datacenters and then add a node to a random datacenter
        print("instance {} fetching catalog".format(self.locust))
        r = self.client.get("/v1/catalog/datacenters", name="get /catalog/datacenters")
        DATACENTER_LIST = r.json()
        node = {}
        node['Datacenter'] = random.choice(DATACENTER_LIST)
        node['Node'] = data_generator(8)
        node['Address'] = ipv4_generator()
        node['Service'] = {}
        node['Service']['Service'] = data_generator(6, string.ascii_lowercase)
        print("instance {} putting node {} in {}".format(self.locust, node['Node'], node['Datacenter']))
        r = self.client.put("/v1/catalog/register", json.dumps(node), name="put /catalog/register")
        return

    @task(1)
    def delete_kv_item(self):
        # Attempt to delete an item in the KV store
        key_name = data_generator(4, "12345678")
        print("instance {} attempting delete of KV item {}".format(self.locust, key_name))
        r = self.client.delete("/v1/kv/{}".format(key_name), name="delete /kv/*")
        return

    @task(0)
    def deregister_node(self):
        # Fetch the list of datacenters and then attempt to deregister a random node in a random datacenter
        print("instance {} fetching catalog".format(self.locust))
        r = self.client.get("/v1/catalog/datacenters", name="get /catalog/datacenters")
        DATACENTER_LIST = r.json()
        node = {}
        node['Datacenter'] = random.choice(DATACENTER_LIST)
        r = self.client.get("/v1/catalog/nodes?dc={}".format(node['Datacenter']), name="get /catalog/nodes?dc=*")
        NODES_LIST = r.json()
        node['Node'] = random.choice(NODES_LIST)['Node']
        # ensure we don't remove anything matching a pre-defined pattern
        if search(PATTERN, node['Node']):
            print("attempt to remove node {} aborted".format(node['Node']))
            return
        print("instance {} attempting to deregister node {}".format(self.locust, node['Node']))
        r = self.client.put("/v1/catalog/deregister", json.dumps(node), name="put /catalog/deregister")
        return


class MyLocust(HttpLocust):
    # define which TaskSet to run
    task_set = TestConsulHTTP
    # minimum time (in ms) between actions, default 1000 for a full second
    min_wait = 3600000
    # maximum time (in ms) between actions, default 1000 for a full second
    max_wait = 3600000
