resource "aws_vpc" "my-vpc" {
  cidr_block           = "${var.vpc_cidr_block}"
  instance_tenancy     = "default"
  enable_dns_support   = "true"
  enable_dns_hostnames = "true"
  enable_classiclink   = "false"

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}"
    )
  )}"
}

# public subnet 1
resource "aws_subnet" "public-1" {
  vpc_id                  = "${aws_vpc.my-vpc.id}"
  cidr_block              = "${var.public_subnet1_cidr}"
  map_public_ip_on_launch = "true"
  availability_zone       = "${element(local.availability_zones, 0)}"

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-public-subnet"
    )
  )}"
}

# define a security group for public subnet
resource "aws_security_group" "public-sg" {
  name        = "${var.vpc_name}-public-sg"
  description = "allow incoming SSH access and ping, allow all outbound"

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  vpc_id = "${aws_vpc.my-vpc.id}"

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-public-sg"
    )
  )}"
}

# Limit prefix
resource "aws_security_group_rule" "allow_prefix_icmp" {
  count = "${var.enable_corp_only_prefix == "true" ? 1 : 0}"

  type            = "ingress"
  from_port       = -1
  to_port         = -1
  protocol        = "icmp"
  prefix_list_ids = "${var.corp_only_prefix}"

  security_group_id = "${aws_security_group.public-sg.id}"
}

resource "aws_security_group_rule" "allow_prefix_ssh" {
  count = "${var.enable_corp_only_prefix == "true" ? 1 : 0}"

  type            = "ingress"
  from_port       = 22
  to_port         = 22
  protocol        = "tcp"
  prefix_list_ids = "${var.corp_only_prefix}"

  security_group_id = "${aws_security_group.public-sg.id}"
}

# Allow all
resource "aws_security_group_rule" "allow_all_icmp" {
  count = "${var.enable_corp_only_prefix != "true" ? 1 : 0}"

  type        = "ingress"
  from_port   = -1
  to_port     = -1
  protocol    = "icmp"
  cidr_blocks = ["0.0.0.0/0"]

  security_group_id = "${aws_security_group.public-sg.id}"
}

resource "aws_security_group_rule" "allow_all_ssh" {
  count = "${var.enable_corp_only_prefix != "true" ? 1 : 0}"

  type        = "ingress"
  from_port   = 22
  to_port     = 22
  protocol    = "tcp"
  cidr_blocks = ["0.0.0.0/0"]

  security_group_id = "${aws_security_group.public-sg.id}"
}

# make internet gateway for public subnet
resource "aws_internet_gateway" "gw" {
  vpc_id = "${aws_vpc.my-vpc.id}"

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-gw"
    )
  )}"
}

# make route table for internet gateway
resource "aws_route_table" "gw_route" {
  vpc_id = "${aws_vpc.my-vpc.id}"

  route {
    cidr_block = "0.0.0.0/0"
    gateway_id = "${aws_internet_gateway.gw.id}"
  }

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-public-gw-route"
    )
  )}"
}

# associate route table with public subnets
resource "aws_route_table_association" "rt-public-1-a" {
  subnet_id      = "${aws_subnet.public-1.id}"
  route_table_id = "${aws_route_table.gw_route.id}"
}

resource "aws_subnet" "private" {
  count = "${length(var.private_subnets_cidr)}"

  vpc_id                  = "${aws_vpc.my-vpc.id}"
  cidr_block              = "${element(var.private_subnets_cidr, count.index)}"
  map_public_ip_on_launch = "false"
  availability_zone       = "${element(local.availability_zones, count.index)}"

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-private-subnet-${count.index}"
    )
  )}"
}

# Define a security group for private subnets (open to everything)
resource "aws_security_group" "sg_private" {
  name        = "${var.vpc_name}-private-sg"
  description = "allow all traffic in and out"

  ingress {
    from_port   = -1
    to_port     = -1
    protocol    = "icmp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  vpc_id = "${aws_vpc.my-vpc.id}"

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-private-sg"
    )
  )}"
}

# associate routes to private networks
resource "aws_route_table_association" "rt_private" {
  count = "${length(local.availability_zones)}"

  subnet_id      = "${element(aws_subnet.private.*.id, count.index)}"
  route_table_id = "${aws_route_table.private.id}"
}

# make elastic ip
resource "aws_eip" "nat" {
  vpc = true

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-eip"
    )
  )}"
}

# make nat gateway in public net 1
resource "aws_nat_gateway" "nat-gw" {
  allocation_id = "${aws_eip.nat.id}"
  subnet_id     = "${aws_subnet.public-1.id}"
  depends_on    = ["aws_internet_gateway.gw"]

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-nat-gw"
    )
  )}"
}

# make route table for private network route to nat gateway
resource "aws_route_table" "private" {
  vpc_id = "${aws_vpc.my-vpc.id}"

  route {
    cidr_block     = "0.0.0.0/0"
    nat_gateway_id = "${aws_nat_gateway.nat-gw.id}"
  }

  tags = "${merge(
    local.common_tags,
    map(
      "Name", "${var.vpc_name}-private-gw"
    )
  )}"
}
