data "template_cloudinit_config" "userdata" {

  part {
    content_type = ""

    content = templatefile("${path.module}/templates/instance-cloudinit.tmpl", {
      dns_region = var.aws_region
      dns_zone   = "${var.env}.consul.live-video.a2z.com"
      role       = var.role
    })
  }

  part {
    content_type = "text/x-shellscript"

    content = templatefile("${path.module}/templates/instance-userdata.sh.tmpl", {
      environment      = local.environment
      consul_dc        = var.aws_region
      role             = var.role
      owner            = "vidcs"
      puppet_server    = "video-puppet.prod.puppet.live-video.a2z.com"
      puppet_ca_server = "video-puppet-ca.prod.puppet.live-video.a2z.com"
    })
  }

}

resource "aws_iam_instance_profile" "instance_profile" {
  name = "${var.role}-${var.aws_region}"
  role = aws_iam_role.role.name
}

# Allow the ec2 instances to assume the ec2 service principal, as well as have sandstorm
# and s3 put access for the existing consul-kv-backup s3 bucket
data "aws_iam_policy_document" "instance_assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_policy" "policy" {
  name   = "${var.role}-${var.aws_region}-backinator-policy"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
        "Sid": "1",
        "Effect": "Allow",
        "Action": ["s3:PutObject"],
        "Resource": "arn:aws:s3:::consul-kv-backup-production/*"
    },
    {
        "Action": ["sts:AssumeRole"],
        "Resource": "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/consul-backinator",
        "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_policy_attachment" "test-attach" {
  name       = "backinator-attachment"
  roles      = [aws_iam_role.role.name]
  policy_arn = aws_iam_policy.policy.arn
}

resource "aws_iam_role" "role" {
  name               = "${var.role}-${var.aws_region}"
  assume_role_policy = data.aws_iam_policy_document.instance_assume_role_policy.json
  tags               = local.common_tags
}

data "aws_iam_policy_document" "cloudwatch_trust_policy_document" {
  statement {
    sid     = "AllowCloudwatchAssume"
    actions = ["sts:AssumeRole"]
    resources = [
      "arn:aws:iam::277437172290:role/CloudwatchAgentMetrics",
      "arn:aws:iam::277437172290:role/CloudwatchAgentLogs",
    ]
  }
}

resource "aws_iam_policy" "cloudwatch_trust_policy" {
  name   = "${var.role}-${var.aws_region}-cw-trust-policy"
  policy = data.aws_iam_policy_document.cloudwatch_trust_policy_document.json
}

resource "aws_iam_role_policy_attachment" "attach_cw_to_consul_role" {
  role       = aws_iam_role.role.name
  policy_arn = aws_iam_policy.cloudwatch_trust_policy.arn
}

data "aws_ami" "xenial_amiid" {
  # Use latest AMI
  most_recent = true
  owners      = ["043714768218"]
  filter {
    name   = "name"
    values = ["devtools-baseami-xenial-consulfree*"]
  }
}

resource "aws_launch_configuration" "cm_launch_template" {
  name_prefix          = "${var.role}_launch_config"
  image_id             = data.aws_ami.xenial_amiid.id
  instance_type        = var.instance_type
  security_groups      = [local.consul_ec2_sg]
  iam_instance_profile = aws_iam_instance_profile.instance_profile.name
  user_data            = data.template_cloudinit_config.userdata.rendered
}

resource "aws_autoscaling_group" "cm_asg" {
  name                = "${var.role}_asg"
  max_size            = var.asg_size
  min_size            = var.asg_size
  health_check_type   = "EC2"
  vpc_zone_identifier = local.consul_vpc_subnets

  launch_configuration = aws_launch_configuration.cm_launch_template.name
  target_group_arns = [
    aws_lb_target_group.consul_nlb_tg.arn,
    aws_lb_target_group.consul_nlb_wan_tg.arn,
    aws_lb_target_group.consul_server_tg.arn,
  ]

  enabled_metrics = ["GroupInServiceInstances", "GroupPendingInstances", "GroupStandbyInstances", "GroupTerminatingInstances"]
  lifecycle {
    create_before_destroy = true
  }

  initial_lifecycle_hook {
    name                    = "lifecycle-launching"
    default_result          = "ABANDON"
    heartbeat_timeout       = 60
    lifecycle_transition    = "autoscaling:EC2_INSTANCE_LAUNCHING"
    notification_target_arn = module.autoscale_dns.autoscale_handling_sns_topic_arn
    role_arn                = module.autoscale_dns.agent_lifecycle_iam_role_arn
  }

  initial_lifecycle_hook {
    name                    = "lifecycle-terminating"
    default_result          = "ABANDON"
    heartbeat_timeout       = 60
    lifecycle_transition    = "autoscaling:EC2_INSTANCE_TERMINATING"
    notification_target_arn = module.autoscale_dns.autoscale_handling_sns_topic_arn
    role_arn                = module.autoscale_dns.agent_lifecycle_iam_role_arn
  }

  dynamic "tag" {
    for_each = merge({
      Name                   = var.role,
      "asg:hostname_pattern" = "${var.role}-#instanceid.${var.aws_region}.${var.env}.consul.live-video.a2z.com@${local.consul_r53_zone_id}",
      "consul-servers-${var.aws_region}" = "consul-servers-${var.aws_region}" },
      local.common_tags
    )

    content {
      key                 = tag.key
      value               = tag.value
      propagate_at_launch = true
    }
  }


}
