module "consul_vpc_endpoint_service" {
  source             = "git@git.xarth.tv:video-coreservices/tf-mod-vpce-service.git?ref=v1.2.2"
  service_id         = "consul-${local.environment_map[var.env]}"
  region             = var.aws_region
  nlb_arn            = aws_lb.consul_vpce_nlb.arn
  nlb_tg_arn         = aws_lb_target_group.consul_vpce_nlb_tg.arn
  alb_arn            = aws_lb.consul_alb.arn
  alb_listener_port  = 443
  allowed_principals = var.vpce_allowed_principals
}

# NLB for the VPCe.
resource "aws_lb" "consul_vpce_nlb" {
  name               = "consul-vpce-nlb"
  internal           = true
  load_balancer_type = "network"
  subnets            = local.consul_vpc_subnets
  idle_timeout       = 300
}

# 443 listener for the NLB.
resource "aws_lb_listener" "consul_vpce_nlb_listener" {
  load_balancer_arn = aws_lb.consul_vpce_nlb.id
  port              = 443
  protocol          = "TLS"
  ssl_policy        = "ELBSecurityPolicy-2016-08"
  certificate_arn   = data.terraform_remote_state.acm_backend.outputs.consul_cert_arn

  default_action {
    target_group_arn = aws_lb_target_group.consul_vpce_nlb_tg.id
    type             = "forward"
  }
}

# Target group that'll get updated by the lambda created by the module.
resource "aws_lb_target_group" "consul_vpce_nlb_tg" {
  name        = "consul-vpce-nlb-tg"
  port        = 443
  protocol    = "TLS"
  vpc_id      = local.consul_vpc
  target_type = "ip"
}


# We want to set up an ALB that exposes port 80 so that consumers
resource "aws_lb" "consul_alb" {
  name                       = "consul-api-${var.aws_region}"
  internal                   = true
  load_balancer_type         = "application"
  security_groups            = [local.consul_ec2_sg]
  subnets                    = local.consul_vpc_subnets
  enable_deletion_protection = true

  tags = local.common_tags
}

# Target group to forward listener traffic to the listening port nginx is configured to listen on in AWS masters
resource "aws_lb_target_group" "consul_server_tg" {
  name     = "consul-api"
  port     = 80
  protocol = "HTTP"
  vpc_id   = local.consul_vpc

  health_check {
    protocol            = "HTTP"
    path                = "/ui/"
    interval            = 10
    healthy_threshold   = 3
    unhealthy_threshold = 3
  }

  tags = local.common_tags
}

## LB listeners

# UI/API (HTTP)
resource "aws_lb_listener" "consul_server_listener" {
  load_balancer_arn = aws_lb.consul_alb.arn
  port              = 80
  protocol          = "HTTP"

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.consul_server_tg.arn
  }
}

data "terraform_remote_state" "acm_backend" {
  backend = "s3"

  config = {
    bucket = var.backend_bucket
    key    = "terraform_bootstrap_acm.tfstate"
    region = var.aws_region
  }
}

# UI/API (HTTPS)
resource "aws_lb_listener" "consul_server_https_listener" {
  load_balancer_arn = aws_lb.consul_alb.arn
  port              = 443
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2016-08"
  certificate_arn   = data.terraform_remote_state.acm_backend.outputs.consul_cert_arn

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.consul_server_tg.arn
  }
}

# Route 53 record for the alb
resource "aws_route53_record" "r53_consul_alb" {
  zone_id = local.consul_r53_zone_id
  name    = "api"
  type    = "A"

  alias {
    name                   = aws_lb.consul_alb.dns_name
    zone_id                = aws_lb.consul_alb.zone_id
    evaluate_target_health = true
  }
}


# We also want to set up an NLB that exposes the serf LAN port so clients that run
# an agent may reference a single DNS name when handling joining/rejoining the DC
resource "aws_lb" "consul_nlb" {
  name                       = "consul-lan-${var.aws_region}"
  internal                   = true
  load_balancer_type         = "network"
  subnets                    = local.consul_vpc_subnets
  enable_deletion_protection = true

  tags = local.common_tags
}

# serf lan port
resource "aws_lb_target_group" "consul_nlb_tg" {
  name     = "consul-lan"
  port     = 8301
  protocol = "TCP_UDP"
  vpc_id   = local.consul_vpc

  health_check {
    protocol            = "TCP"
    interval            = 10
    healthy_threshold   = 3
    unhealthy_threshold = 3
  }

  tags = local.common_tags
}

resource "aws_lb_listener" "consul_nlb_listener" {
  load_balancer_arn = aws_lb.consul_nlb.arn
  port              = 8301
  protocol          = "TCP"

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.consul_nlb_tg.arn
  }
}

# Route 53 record for the nlb
resource "aws_route53_record" "r53_consul_nlb" {
  zone_id = local.consul_r53_zone_id
  name    = "lan"
  type    = "A"

  alias {
    name                   = aws_lb.consul_nlb.dns_name
    zone_id                = aws_lb.consul_nlb.zone_id
    evaluate_target_health = true
  }
}

# wan port
resource "aws_lb_target_group" "consul_nlb_wan_tg" {
  name     = "consul-wan"
  port     = 8302
  protocol = "TCP_UDP"
  vpc_id   = local.consul_vpc

  health_check {
    protocol            = "TCP"
    interval            = 10
    healthy_threshold   = 3
    unhealthy_threshold = 3
  }

  tags = local.common_tags
}

resource "aws_lb_listener" "consul_nlb_wan_listener" {
  load_balancer_arn = aws_lb.consul_nlb.arn
  port              = 8302
  protocol          = "TCP"

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.consul_nlb_wan_tg.arn
  }
}

# Route 53 record for the wan nlb
# This is mostly a vanity entry since there's basically a same entry for lan
resource "aws_route53_record" "r53_consul_wan_nlb" {
  zone_id = local.consul_r53_zone_id
  name    = "wan"
  type    = "A"

  alias {
    name                   = aws_lb.consul_nlb.dns_name
    zone_id                = aws_lb.consul_nlb.zone_id
    evaluate_target_health = true
  }
}
