// Setup a VPC resolver for internal twitch DNS
// https://wiki.twitch.com/display/PS/How+to+Configure+a+VPC+Resolver+for+Internal+Twitch+DNS

locals {
  // https://docs.aws.amazon.com/vpc/latest/userguide/vpc-dns.html
  domain_name = "${var.region}.compute.internal"

  // Baremetal anycast DNS Resolvers
  twitch_dns_servers = ["10.254.0.3", "10.254.0.8"]

  // private subnet IDs
  private_subnet_ids = module.defaults.private_subnet_ids
}

resource "aws_security_group" "dns_to_justin_tv_resolver" {
  name        = "r53_to_justin_tv_resolvers"
  description = "outbound resolver SG to baremetal DNS servers"
  vpc_id      = module.defaults.vpc_id

  egress {
    from_port   = 53
    to_port     = 53
    protocol    = "udp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  egress {
    from_port   = 53
    to_port     = 53
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = merge({ Name = "r53_to_justin_tv_resolvers" }, local.common_tags)
}

resource "aws_route53_resolver_endpoint" "vpc_outbound_dns" {
  name      = "vpc_outbound_dns"
  direction = "OUTBOUND"

  security_group_ids = [aws_security_group.dns_to_justin_tv_resolver.id]

  dynamic "ip_address" {
    for_each = local.private_subnet_ids

    content {
      subnet_id = ip_address.value
    }
  }
}

# Setup rule to forward .justin.tv domains to baremetal servers.
resource "aws_route53_resolver_rule" "fwd_justin_tv" {
  name                 = "fwd_justin_tv"
  domain_name          = "justin.tv"
  rule_type            = "FORWARD"
  resolver_endpoint_id = aws_route53_resolver_endpoint.vpc_outbound_dns.id

  dynamic "target_ip" {
    for_each = local.twitch_dns_servers

    content {
      ip = target_ip.value
    }
  }
}

resource "aws_route53_resolver_rule_association" "fwd_justin_tv" {
  resolver_rule_id = aws_route53_resolver_rule.fwd_justin_tv.id
  vpc_id           = module.defaults.vpc_id
}
