locals {
  # Bucket name is set here as module.dns_valiate_lambda_deploy_stack relies on a s3 bucket name count
  # that could not be computed on initial bootstrap
  s3_bucket = "dns-validator-codesource-bucket-${var.env}-${var.region}"
  src_dir   = "${path.module}/lambda"
  build_dir = "${path.module}/build"
}

resource "aws_s3_bucket" "service_codesource_bucket" {
  bucket = local.s3_bucket
  acl    = "private"

  versioning {
    enabled = true
  }
}

# Ensure new archive is only uploaded on python code changes
resource "random_uuid" "lambda_src_hash" {
  keepers = {
    for filename in setunion(
      fileset("${local.src_dir}", "*.py"),
      fileset("${local.src_dir}", "requirements.txt"),
    ) :
    filename => filemd5("${local.src_dir}/${filename}")
  }
}

data "archive_file" "lambda_code" {
  type        = "zip"
  source_dir  = "${local.src_dir}"
  output_path = "${local.build_dir}/${random_uuid.lambda_src_hash.result}.zip"
  excludes    = [".pytest_cache", "requirements-dev.txt", "tests"]
}

resource "aws_s3_bucket_object" "lambda_zip_to_codesource" {
  bucket     = local.s3_bucket
  key        = "main/SourceArtifact.zip"
  source     = "${local.build_dir}/${random_uuid.lambda_src_hash.result}.zip"
  depends_on = [aws_s3_bucket.service_codesource_bucket]
}

module "dns_validate_lambda_deploy_stack" {
  source                      = "git@git.xarth.tv:video-coreservices/tf-mod-lambda-pipeline.git?ref=v1.1.0"
  env                         = var.env
  region                      = var.region
  repo_src                    = local.common_tags["Source"]
  service_id                  = "dns-validator"
  lambda_function_name        = "dns_validate_lambda"
  codesource_bucket_id        = local.s3_bucket
  codesource_bucket_subfolder = "main"
  vpc_id                      = module.shared.vpc_id
  security_group_ids          = [module.shared.sg_twitch_subnet_id]
  subnet_ids                  = module.shared.private_subnet_ids
}

# Slack notifications to vidcs-ddns-deploy channel
module "slack_sns" {
  source             = "git@git.xarth.tv:video/chatbot-terraform.git//modules/sns?ref=v1.0.0"
  codepipeline_arn   = module.dns_validate_lambda_deploy_stack.codepipeline_arn
  codepipeline_name  = module.dns_validate_lambda_deploy_stack.codepipeline_name
  configuration_name = "${module.dns_validate_lambda_deploy_stack.codepipeline_name}-slack-chatbot"
}

module "slack_chatbot" {
  source           = "git@git.xarth.tv:video/chatbot-terraform.git//modules/chatbot?ref=v1.0.0"
  slack_channel_id = "C02BTS4HLAH" # vidcs-ddns-deploy
  sns_topic_arns = [
    module.slack_sns.chatbot_sns_topic,
  ]
}
