# SNS topics that will receive message from bare metal DHCP clients
resource "aws_sns_topic" "crud_updater_topic" {
  name = "hls-dns-crud-updater"
}

# Create subscription to new dns validator lambda
resource "aws_sns_topic_subscription" "sqs_dns_validate_subscription" {
  topic_arn = aws_sns_topic.crud_updater_topic.arn
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.dns_validate_queue.arn
}

data "aws_iam_policy_document" "crud_sns_policy_document" {
  statement {
    sid = "DefaultPolicy"

    actions = [
      "SNS:GetTopicAttributes",
      "SNS:SetTopicAttributes",
      "SNS:AddPermission",
      "SNS:RemovePermission",
      "SNS:DeleteTopic",
      "SNS:Subscribe",
      "SNS:ListSubscriptionsByTopic",
      "SNS:Publish",
      "SNS:Receive",
    ]
    effect = "Allow"

    principals {
      type        = "AWS"
      identifiers = ["*"]
    }

    resources = [
      aws_sns_topic.crud_updater_topic.arn,
    ]

    condition {
      test     = "StringEquals"
      variable = "AWS:SourceOwner"
      values   = [data.aws_caller_identity.caller.account_id]
    }
  }
}

resource "aws_sns_topic_policy" "crud_topic_policy" {
  arn    = aws_sns_topic.crud_updater_topic.arn
  policy = data.aws_iam_policy_document.crud_sns_policy_document.json
}

## OUTBOUND
# SNS topics that will receive message for validated machines
resource "aws_sns_topic" "crud_sanitized_topic" {
  name = "ddns-verified-machines"
}

data "aws_iam_policy_document" "crud_sanitized_sns_policy_document" {
  statement {
    sid = "DefaultPolicy"

    actions = [
      "SNS:GetTopicAttributes",
      "SNS:SetTopicAttributes",
      "SNS:AddPermission",
      "SNS:RemovePermission",
      "SNS:DeleteTopic",
      "SNS:Subscribe",
      "SNS:ListSubscriptionsByTopic",
      "SNS:Publish",
      "SNS:Receive",
    ]
    effect = "Allow"

    principals {
      type        = "AWS"
      identifiers = ["*"]
    }

    resources = [
      aws_sns_topic.crud_sanitized_topic.arn,
    ]

    condition {
      test     = "StringEquals"
      variable = "AWS:SourceOwner"
      values   = [data.aws_caller_identity.caller.account_id]
    }
  }

  # Gatekeeper policy:
  # Permit other accounts to subscribe to ddns-verified-machines SNS Topic
  statement {
    sid = "AllowXAccountAccess"

    actions = [
      "SNS:Subscribe",
      "SNS:Receive",
    ]

    effect = "Allow"

    principals {
      type        = "AWS"
      identifiers = local.cross_account_subs[var.env][var.region]
    }

    resources = [
      aws_sns_topic.crud_sanitized_topic.arn,
    ]
  }
}

resource "aws_sns_topic_policy" "crud_sanitized_topic_policy" {
  arn    = aws_sns_topic.crud_sanitized_topic.arn
  policy = data.aws_iam_policy_document.crud_sanitized_sns_policy_document.json
}
