## Setup sanitized workflow queue for DNS validator

# the queue that will receive cross account SNS notifications
resource "aws_sqs_queue" "dns_validate_queue" {
  name                       = "dns-validate-queue"
  visibility_timeout_seconds = 900 # 15 mins
  delay_seconds              = 60
  tags                       = local.common_tags
  message_retention_seconds  = 86400 # one day
  redrive_policy             = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.dns_validate_deadletter.arn}\",\"maxReceiveCount\":3}"
}

# the deadletter queue is where failed to process messages (processed unsuccessfully > 3 times)
# get sent to for debugging
resource "aws_sqs_queue" "dns_validate_deadletter" {
  name                      = "dns-validate-deadletter"
  message_retention_seconds = 604800 # seven days
  tags                      = local.common_tags
}

# Allow sns topic to send messages
resource "aws_sqs_queue_policy" "allow_sns_to_send_to_dns_validate" {
  queue_url = aws_sqs_queue.dns_validate_queue.id
  policy    = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "SQSPolicy",
  "Statement": [
    {
      "Sid": "AllowSNSToSend",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "sqs:SendMessage",
      "Resource": "${aws_sqs_queue.dns_validate_queue.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${aws_sns_topic.crud_updater_topic.arn}"
        }
      }
    },
    {
      "Sid": "AllowLambdaActions",
      "Effect": "Allow",
      "Principal": "*",
      "Action": [
        "sqs:ReceiveMessage",
        "sqs:DeleteMessage",
        "sqs:GetQueueAttributes"
      ],
      "Resource": "${aws_sqs_queue.dns_validate_queue.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${aws_lambda_function.dns_validate_lambda.arn}"
        }
      }
    }
  ]
}
POLICY
}

# the queue that the lambda will send to if there's a DNS mismatch or SERVFAIL
resource "aws_sqs_queue" "dns_validate_delay_queue" {
  name                       = "dns-validate-delay-queue"
  visibility_timeout_seconds = 120 # 2 mins, timeout has to be longer than lambda timeout (60 seconds)
  delay_seconds              = 900 # 15 mins
  tags                       = local.common_tags
  message_retention_seconds  = 86400 # one day
  redrive_policy             = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.dns_validate_delay_deadletter.arn}\",\"maxReceiveCount\":1}"
}

# the deadletter queue is where failed to process messages (processed unsuccessfully > 3 times)
# get sent to for manual debugging
resource "aws_sqs_queue" "dns_validate_delay_deadletter" {
  name                      = "dns-validate-delay-deadletter"
  message_retention_seconds = 604800 # seven days
  tags                      = local.common_tags
}

# Allows the lambda function to send messages to the retry delay queue as well as the lambda to do its event mapping stuff
resource "aws_sqs_queue_policy" "allow_dns_validate_lambda_to_send_to_retry" {
  queue_url = aws_sqs_queue.dns_validate_delay_queue.id
  policy    = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "sqspolicy",
  "Statement": [
    {
      "Sid": "AllowLambdaActions",
      "Effect": "Allow",
      "Principal": "*",
      "Action": [
        "sqs:SendMessage",
        "sqs:ReceiveMessage",
        "sqs:DeleteMessage",
        "sqs:GetQueueAttributes"
      ],
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${aws_lambda_function.dns_validate_lambda.arn}"
        }
      }
    }
  ]
}
POLICY
}
