#!/usr/bin/env bash

set -eo pipefail

export AWS_STS_REGIONAL_ENDPOINTS=regional
export AWS_SDK_LOAD_CONFIG=true

# twitch-vidcs+pop-delegation-control-dns
AWS_ACCOUNT_ID=362079722421

usage() {
    cat <<EOF >&2
Usage:
  $0 -p <pop> <command> [args...]
  <pop>
    The name of the pop. E.g. pdx05. This must be a valid pop.
  <command> [args ...]
    In general, these are command and args that are passed to terraform binary as
    arguments
    E.g.
      $0 plan
      gets passed to terraform as
      terraform plan
EOF
    exit 1
}

err() {
    echo "ERROR: $*" >&2
}

requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && usage )
}

tf_init() {
    terraform init -reconfigure -backend-config=<(cat <<EOF
key = "pop/${pop}.tfstate"
EOF
)
    terraform get -update
}

validate_account() {
    if ! [[ "$AWS_ACCOUNT_ID" == "$(aws sts get-caller-identity | jq -r .Account)" ]]; then
        err "$(cat <<EOF
  Your current credentials do not match account '${AWS_ACCOUNT_ID}' that manages this environment and region
  Please double check that you are using the correct credentials for the correct account.
EOF
)"
        usage
    fi
}

while getopts "hp:" option; do
  case "$option" in
    p)
      pop="$OPTARG"
      ;;
    h)
      usage
      ;;
    *)
      ;;
  esac
done

# Make sure everything is set
[[ -z "$pop" ]] && usage

requirements
validate_account

shift $((OPTIND-1))

case "$1" in
    init)
        tf_init
        ;;
    plan|apply)
        tf_init
        exec terraform "$@" \
            -var="pop=${pop}"
        ;;
    import)
        tf_init
        shift
        exec terraform import -var="pop=${pop}" "$@"
        ;;
    *)
        exec terraform "$@"
        ;;
esac
