terraform {
  backend "s3" {}
}

module "variables" {
  source = "./vendor/modules/twitch-video-ops-defaults/current"
  region = "${var.region}"
}

# Fetch latest AMI
data "aws_ami" "base_ami" {
  most_recent = true
  owners      = ["043714768218"]

  filter {
    name   = "name"
    values = ["devtools-baseami-xenial-*"]
  }
}

locals {
  ami_id = "${var.ami_id == "" ? data.aws_ami.base_ami.id : var.ami_id}"
}

module "instances" {
  source                 = "./vendor/modules/aws-instances"
  account                = "${module.variables.account_name}"
  region                 = "${var.region}"
  environment            = "${local.environment}"
  vpc_security_group_ids = ["${module.variables.sg_twitch_subnet_id}"]
  subnet_ids             = ["${module.variables.private_subnet_ids}"]
  service                = "ganglia"
  role                   = "ganglia"
  ami                    = "${local.ami_id}"
  instance_type          = "${var.instance_type}"
  puppet                 = "puppet"
  count                  = "${var.instance_count}"
  owner                  = "vidcs"
  r53_zone_id            = "${module.variables.r53_zone_id}"
  r53_zone_name          = "${module.variables.r53_zone_name}"

  additional_run_cmds = [
    "echo \"$(date +'%F %T %Z') -- Removing amazon-ssm-agent\"",
    "bash -c 'while ! snap remove amazon-ssm-agent; do echo \"Unable to remove amazon-ssm agent. Retrying in 10 seconds.\"; sleep 10; done'",
    "echo \"$(date +'%F %T %Z') -- RAIDing (mirroring) local NVMe disks\"",
    "mkdir -p /var/lib/ganglia",
    "echo \"LABEL=ganglia /var/lib/ganglia xfs auto,rw,noexec,nosuid,noatime,nobarrier,logbufs=8,logbsize=256k 0 0\" >> /etc/fstab",
    "mdadm --create --verbose /dev/md0 --level=1 --name=ganglia --raid-devices=2 /dev/nvme0n1 /dev/nvme1n1 --metadata=0.90",
    "echo \"$(date +'%F %T %Z') -- formatting /dev/md0 (raid 1) as XFS\"",
    "mkfs.xfs -L ganglia /dev/md0",
    "mount /var/lib/ganglia",
    "rm -f /etc/facter/facts.d/disable_consul.txt",
  ]
}

# ENI attachment
module "ganglia_enis" {
  source                 = "./vendor/modules/eni-creation"
  account                = "${module.variables.account_name}"
  region                 = "${var.region}"
  eni_name               = "ganglia"
  eni_count              = "${var.instance_count}"
  vpc_security_group_ids = ["${module.variables.sg_twitch_subnet_id}"]
  subnet_ids             = ["${module.variables.private_subnet_ids}"]
}

resource "aws_network_interface_attachment" "ganglia_eni_attachment" {
  count = "${var.instance_count}"

  instance_id          = "${element(module.instances.instance_ids, count.index)}"
  network_interface_id = "${element(module.ganglia_enis.ids, count.index)}"
  device_index         = 1
}

data "aws_network_interface" "eni_ip" {
  count = "${var.instance_count}"

  id = "${element(module.ganglia_enis.ids, count.index)}"
}

resource "aws_route53_record" "eni_alias" {
  count = "${var.instance_count}"

  zone_id = "${module.variables.r53_zone_id}"
  name    = "gmond-${count.index + 1}"
  type    = "A"
  ttl     = "300"
  records = ["${element(data.aws_network_interface.eni_ip.*.private_ip, count.index)}"]
}

# ALB attachment with SSL
data "aws_acm_certificate" "ssl_cert" {
  domain      = "${var.ssl_domain}"
  statuses    = ["ISSUED"]
  most_recent = true
}

module "lb" {
  source = "./vendor/modules/lb"

  owner               = "video-coreservices@twitch.tv"
  environment         = "${var.env}"
  vpc_id              = "${module.variables.vpc_id}"
  service             = "ganglia"
  ssl_certificate_arn = "${data.aws_acm_certificate.ssl_cert.arn}"
  subnets             = ["${module.variables.private_subnet_ids}"]
  security_groups     = ["${module.variables.sg_twitch_subnet_id}"]
  instance_ids        = ["${module.instances.instance_ids}"]
  instance_count      = "${var.instance_count}"
  r53_zone_id         = "${module.variables.r53_zone_id}"
  healthcheck_matcher = "401"
}
