###########################################################
# Manage twitch-pop-hls-dns permissions for DNS changes
# For console/sts access, permit posix groups via Isengard
#   https://isengard.amazon.com/roles/393787499934
###########################################################

locals {
  role_name = "twitch-pop-hls-dns"
}

resource "aws_iam_policy" "pop_hls_crud_policy" {
  name        = "twitch-pop-hls-dns-route53"
  description = "Permit DNS record changes on all Hosted Zones with the exception of hls.live-video.net hosted zone"

  policy = templatefile("${path.module}/templates/twitch-pop-hls-dns-policy.json.tmpl", {
    restricted_zone = module.primary.primary.zone.id
  })
}

resource "aws_iam_role_policy_attachment" "pop_dns_crud_attachment" {
  role       = local.role_name
  policy_arn = aws_iam_policy.pop_hls_crud_policy.arn
}

resource "aws_iam_role_policy_attachment" "pop_dns_readonly_attachment" {
  role       = local.role_name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}

###########################################################
# Permit route53 service to log to any /aws/route53 log group
###########################################################

# cloudWatch log resource policy to allow Route53 to write logs
# to any log group under /aws/route53/hls-live-video-net-query-logs
data "aws_iam_policy_document" "r53_query_logging_policy" {
  statement {
    actions = [
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]
    resources = ["arn:aws:logs:us-east-1:${data.aws_caller_identity.current.account_id}:log-group:${local.r53_query_log_group}/*"]

    principals {
      identifiers = ["route53.amazonaws.com"]
      type        = "Service"
    }
  }
}

resource "aws_cloudwatch_log_resource_policy" "r53_query_logging_policy" {
  policy_document = data.aws_iam_policy_document.r53_query_logging_policy.json
  policy_name     = "route53-query-logging-policy"
}

################################################
# Permit dns-updater lambda to operate changes
################################################
data "aws_iam_policy_document" "crud_updater" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        "arn:aws:iam::873126202975:role/hls-lv-net-dns-updater-lambda-role", # twitch-vidcs+hls-lv-net-edgeupdate-prod-us-east-2
        "arn:aws:iam::398750410825:role/hls-lv-net-dns-updater-lambda-role", # twitch-vidcs+hls-lv-net-edgeupdate-prod-us-west-2
      ]
    }
  }
}

resource "aws_iam_role" "crud_updater" {
  name               = "hls-crud-lambda"
  assume_role_policy = data.aws_iam_policy_document.crud_updater.json
}

resource "aws_iam_role_policy_attachment" "pop_dns_crud_updater_attachment" {
  role       = aws_iam_role.crud_updater.name
  policy_arn = aws_iam_policy.pop_hls_crud_policy.arn
}

resource "aws_iam_role_policy_attachment" "pop_dns_crud_updater_readonly_attachment" {
  role       = aws_iam_role.crud_updater.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}


################################################
# Permit ivs_stats_uri_rotation_staging
################################################
data "aws_iam_policy_document" "ivs_stats_uri_rotation_staging" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        # twitch-video-intelligence+stats-endpoint@amazon.com
        "arn:aws:iam::792403465518:role/ivs-stats-uri-rotation-staging-role",
      ]
    }
  }
}

resource "aws_iam_policy" "ivs_stats_uri_rotation_staging" {
  name        = "ivs-stats-uri-rotation-staging"
  description = "Permit DNS record changes on specific Hosted Zones for IVS stats URI Rotation Staging"

  policy = templatefile("${path.module}/templates/twitch-pop-hls-dns-policy-limited.json.tmpl", {
    hosted_zones = [
      "Z1ZDYVSFKGTP62", # sjc02.hls.live-video.net
    ]
  })
}

resource "aws_iam_role" "ivs_stats_uri_rotation_staging" {
  name               = "ivs-stats-uri-rotation-staging"
  assume_role_policy = data.aws_iam_policy_document.ivs_stats_uri_rotation_staging.json
}

resource "aws_iam_role_policy_attachment" "ivs_stats_uri_rotation_staging_attachment" {
  role       = aws_iam_role.ivs_stats_uri_rotation_staging.name
  policy_arn = aws_iam_policy.ivs_stats_uri_rotation_staging.arn
}

resource "aws_iam_role_policy_attachment" "ivs_stats_uri_rotation_staging_readonly_attachment" {
  role       = aws_iam_role.ivs_stats_uri_rotation_staging.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}

################################################
# Permit management of pop subdomain delegation
# from video-coreservices/tf-io-dns-pop-delegation
################################################

module "pop_delegation" {
  source           = "git@git.xarth.tv:video-coreservices/tf-mod-live-video-dns-delegation.git//modules/pop_delegation_role?ref=v1.0.10"
  primary_zone_arn = module.primary.primary.zone.id
}


################################################
# Permit ivs_stats_uri_rotation_prod
################################################
data "aws_iam_policy_document" "ivs_stats_uri_rotation_prod" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        # twitch-video-intelligence+stats-endpoint@amazon.com
        "arn:aws:iam::792403465518:role/ivs-stats-uri-rotation-role",
      ]
    }
  }
}

resource "aws_iam_policy" "ivs_stats_uri_rotation_prod" {
  name        = "ivs-stats-uri-rotation-prod"
  description = "Permit DNS record changes on specific Hosted Zones for IVS stats URI Rotation Prod"

  policy = templatefile("${path.module}/templates/twitch-pop-hls-dns-policy-limited.json.tmpl", {
    hosted_zones = [
      "Z20RL0ZPY2E9Y2", # sjc05.hls.live-video.net
    ]
  })
}

resource "aws_iam_role" "ivs_stats_uri_rotation_prod" {
  name               = "ivs-stats-uri-rotation-prod"
  assume_role_policy = data.aws_iam_policy_document.ivs_stats_uri_rotation_prod.json
}

resource "aws_iam_role_policy_attachment" "ivs_stats_uri_rotation_prod_attachment" {
  role       = aws_iam_role.ivs_stats_uri_rotation_prod.name
  policy_arn = aws_iam_policy.ivs_stats_uri_rotation_prod.arn
}

resource "aws_iam_role_policy_attachment" "ivs_stats_uri_rotation_prod_readonly_attachment" {
  role       = aws_iam_role.ivs_stats_uri_rotation_prod.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}


output "iam_roles_external" {
  description = "External IAM roles that has access to this account"
  value = [
    aws_iam_role.crud_updater.arn,
    aws_iam_role.ivs_stats_uri_rotation_staging.arn,
    aws_iam_role.ivs_stats_uri_rotation_prod.arn,
  ]
}
