###########################################################
# Manage twitch-pop-hls-ttvnw-net-dns permissions for DNS changes
# For console/sts access, permit posix groups via Isengard
#   https://isengard.amazon.com/roles/707969931081
###########################################################

locals {
  role_name = "twitch-pop-hls-ttvnw-net-dns"
}

resource "aws_iam_policy" "pop_dns_crud_policy" {
  name        = "twitch-pop-hls-ttvnw-net-dns-route53"
  description = "Permit DNS record changes on all Hosted Zones with the exception of hls.ttvnw.net hosted zone"

  policy = templatefile("${path.module}/templates/twitch-pop-dns-policy.json.tmpl", {
    restricted_zone = module.primary.primary.zone.id
  })
}

resource "aws_iam_role_policy_attachment" "pop_dns_crud_attachment" {
  role       = local.role_name
  policy_arn = aws_iam_policy.pop_dns_crud_policy.arn
}

resource "aws_iam_role_policy_attachment" "pop_dns_readonly_attachment" {
  role       = local.role_name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}


################################################
# Permit management of pop subdomain delegation
# from video-coreservices/tf-io-dns-pop-delegation
################################################

module "pop_delegation" {
  source           = "git@git.xarth.tv:video-coreservices/tf-mod-live-video-dns-delegation.git//modules/pop_delegation_role?ref=v1.0.10"
  primary_zone_arn = module.primary.primary.zone.id
}


############################
# Permit spade URI rotation
############################
data "aws_iam_policy_document" "spade_uri_rotation_prod" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        # twitch-aws-owners+science@amazon.com
        "arn:aws:iam::673385534282:role/science-jobs-prod",
      ]
    }
  }
}

resource "aws_iam_policy" "spade_uri_rotation_prod" {
  name        = "spade-uri-rotation-prod"
  description = "Permit DNS record changes on specific Hosted Zones for IVS stats URI Rotation Prod"

  policy = templatefile("${path.module}/templates/twitch-pop-hls-dns-policy-limited.json.tmpl", {
    hosted_zones = [
      "Z103992925GN0YO5EIIVZ", # pdx01.abs.hls.ttvnw.net
    ]
  })
}

resource "aws_iam_role" "spade_uri_rotation_prod" {
  name               = "spade-uri-rotation-prod"
  assume_role_policy = data.aws_iam_policy_document.spade_uri_rotation_prod.json
}

resource "aws_iam_role_policy_attachment" "spade_uri_rotation_prod_attachment" {
  role       = aws_iam_role.spade_uri_rotation_prod.name
  policy_arn = aws_iam_policy.spade_uri_rotation_prod.arn
}

resource "aws_iam_role_policy_attachment" "spade_uri_rotation_prod_readonly_attachment" {
  role       = aws_iam_role.spade_uri_rotation_prod.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}

##############################################################################
# Permit staging account to perform R53 operations on the staging subdomains.
##############################################################################
data "aws_iam_policy_document" "staging_dns_updater" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        # twitch-vidcs+hls-ttvnw-net-edgeupdate-staging-usw2@amazon.com
        "arn:aws:iam::007812478044:role/hls-ttvnw-net-dns-updater-lambda-role",
      ]
    }
  }
}

resource "aws_iam_policy" "staging_dns_updater" {
  name        = "hls-ttvnw-net-dns-updater-lambda-role-staging"
  description = "Permit DNS record changes on specific Hosted Zones for the staging hls.ttvnw.net DNS updater account."

  policy = templatefile("${path.module}/templates/twitch-pop-hls-dns-policy-limited.json.tmpl", {
    hosted_zones = [
      "Z0943675TKYGWX7WF0NO",  # staging.abs.hls.ttvnw.net
      "Z094042621JYPJQF1XJAL", # staging.no-abs.hls.ttvnw.net
    ]
  })
}

resource "aws_iam_role" "staging_dns_updater" {
  name               = "hls-ttvnw-net-dns-updater-lambda-role-staging"
  assume_role_policy = data.aws_iam_policy_document.staging_dns_updater.json
}

resource "aws_iam_role_policy_attachment" "staging_dns_updater_attachment" {
  role       = aws_iam_role.staging_dns_updater.name
  policy_arn = aws_iam_policy.staging_dns_updater.arn
}

resource "aws_iam_role_policy_attachment" "staging_dns_updater_readonly_attachment" {
  role       = aws_iam_role.staging_dns_updater.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}

###################################################################################
# Permit prod account to perform R53 operations on all *.hls.ttvnw.net subdomains.
###################################################################################
data "aws_iam_policy_document" "prod_dns_updater" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        "arn:aws:iam::599087406594:role/hls-ttvnw-net-dns-updater-lambda-role", # twitch-vidcs+hls-ttvnw-net-edgeupdate-prod-us-east-2
        "arn:aws:iam::102890704200:role/hls-ttvnw-net-dns-updater-lambda-role", # twitch-vidcs+hls-ttvnw-net-edgeupdate-prod-us-west-2
      ]
    }
  }
}

resource "aws_iam_role" "prod_dns_updater" {
  name               = "hls-ttvnw-net-dns-updater-lambda-role"
  assume_role_policy = data.aws_iam_policy_document.prod_dns_updater.json
}

resource "aws_iam_role_policy_attachment" "prod_dns_updater_attachment" {
  role       = aws_iam_role.prod_dns_updater.name
  policy_arn = aws_iam_policy.pop_dns_crud_policy.arn
}

resource "aws_iam_role_policy_attachment" "prod_dns_updater_readonly_attachment" {
  role       = aws_iam_role.prod_dns_updater.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}
