data "template_cloudinit_config" "userdata" {
  gzip          = false
  base64_encode = false

  part {
    filename     = "init.cfg"
    content_type = "text/cloud-config"

    content = templatefile("${path.module}/templates/cloudinit.yaml.tmpl", {
      service    = "video-infoblox-snmp-collector"
      dns_region = var.region
      dns_zone   = module.variables.r53_zone_name
    })
  }

  part {
    content_type = "text/x-shellscript"

    content = templatefile("${path.module}/templates/startup.sh.tmpl", {
      environment = local.environment
      owner       = "vidcs"
      service     = "video-infoblox-snmp-collector"
      dns_region  = var.region
      dns_zone    = module.variables.r53_zone_name
    })
  }
}

resource "aws_iam_instance_profile" "instance_profile" {
  name = "video-infoblox-snmp-collector-${var.env}"
  role = aws_iam_role.role.name
}

data "aws_iam_policy_document" "instance_assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "role" {
  name               = "video-infoblox-snmp-collector-${var.env}"
  assume_role_policy = data.aws_iam_policy_document.instance_assume_role_policy.json
}


data "aws_ami" "ubuntu" {
  most_recent = true

  filter {
    name   = "name"
    values = ["ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64-server-*"]
  }

  filter {
    name   = "virtualization-type"
    values = ["hvm"]
  }

  owners = ["099720109477"] # Canonical
}

resource "aws_launch_configuration" "launch_conf" {
  name_prefix          = "video-infoblox-snmp-collector_launch_config"
  image_id             = data.aws_ami.ubuntu.id
  instance_type        = var.instance_type
  iam_instance_profile = aws_iam_instance_profile.instance_profile.name
  security_groups = [
    module.variables.sg_twitch_subnet_id,
  ]
  user_data = data.template_cloudinit_config.userdata.rendered

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "asg" {
  name                 = "video-infoblox-snmp-collector_asg"
  vpc_zone_identifier  = module.variables.private_subnet_ids
  max_size             = 1
  min_size             = 1
  health_check_type    = "EC2"
  launch_configuration = aws_launch_configuration.launch_conf.name

  tags = [
    {
      key                 = "Name"
      value               = "video-infoblox-snmp-collector"
      propagate_at_launch = true
    },
    {
      key                 = "Owner"
      value               = lookup(local.common_tags, "Owner")
      propagate_at_launch = true
    },
    {
      key                 = "Environment"
      value               = lookup(local.common_tags, "Environment")
      propagate_at_launch = true
    },
    {
      key                 = "Service"
      value               = lookup(local.common_tags, "Service")
      propagate_at_launch = true
    },
    {
      key                 = "Source"
      value               = lookup(local.common_tags, "Source")
      propagate_at_launch = true
    }
  ]

}

resource "aws_iam_role_policy_attachment" "cw_agent_metrics" {
  role       = aws_iam_role.role.id
  policy_arn = "arn:aws:iam::aws:policy/CloudWatchAgentServerPolicy"
}

resource "aws_iam_role_policy_attachment" "ssm_console_access" {
  role       = aws_iam_role.role.id
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}
