resource "aws_iam_user" "user" {
  name = "${local.service_name}-user"
  path = "/"
}

resource "aws_iam_role" "role" {
  name               = "${local.service_name}-role"
  assume_role_policy = data.aws_iam_policy_document.assume_role_policy.json
  tags               = local.tags
}

data "aws_iam_policy_document" "assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["lambda.amazonaws.com"]
    }
  }
}

resource "aws_iam_policy" "policy" {
  name   = local.service_name
  path   = "/"
  policy = data.aws_iam_policy_document.execute.json
}

data "aws_iam_policy_document" "execute" {
  # allow lambda to write to Cloudwatch
  statement {
    actions = [
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents"
    ]

    resources = [
       "arn:aws:logs:us-west-2:277437172290:log-group:/aws/lambda/infoblox-backup:*",
    ]
  }

  # For bucket operations
  statement {
    actions = [
      "s3:*"
    ]

    resources = [
       "arn:aws:s3:::${var.output_bucket}",
       "arn:aws:s3:::${var.output_bucket}/*"
    ]
  }

  # For all VPC actions
  statement {
    actions = [
      "ec2:DescribeInstances",
      "ec2:AttachNetworkInterface",
      "ec2:CreateNetworkInterface",
      "ec2:DescribeNetworkInterfaces",
      "ec2:DeleteNetworkInterface",
    ]
    resources = [
      "*",
    ]
  }

  # For access to SSM
  statement {
    actions = [
      "ssm:DescribeParameters",
      "ssm:PutParameter",
      "ssm:DeleteParameter",
      "ssm:GetParameterHistory",
      "ssm:GetParametersByPath",
      "ssm:GetParameters",
      "ssm:GetParameter",
      "ssm:DeleteParameters",
    ]

    resources = [
      "arn:aws:ssm:*:277437172290:parameter/infoblox/backups"
    ]
  }

}

resource "aws_iam_policy_attachment" "policy_attachment" {
  name       = "global-attachment"
  users      = ["${aws_iam_user.user.name}"]
  roles      = ["${aws_iam_role.role.name}"]
  policy_arn = aws_iam_policy.policy.arn
}

