import sys
sys.path.insert(0, 'package/')  # tell python to include the package directory
import boto3
import functools
import json
import os
import requests
from datetime import datetime

s3 = boto3.resource('s3')
bucket = s3.Bucket(os.environ['BACKUP_BUCKET'])
INFOBLOX_GRIDMASTER = os.environ['INFOBLOX_GRIDMASTER']
REGION = os.environ['REGION']
ssm = boto3.client('ssm', REGION)


def get_parameters():
    response = ssm.get_parameters(
        Names=['/infoblox/backups'], WithDecryption=True
    )
    for parameter in response['Parameters']:
        return parameter['Value']


def log_event_on_error(handler):
    """
    decorator that prints trigger event if handler throws an exception
    """
    @functools.wraps(handler)
    def wrapper(event, context):
        try:
            return handler(event, context)
        except Exception:
            print('event = %r' % event)
            raise
    return wrapper


@log_event_on_error
def lambda_handler(event, context):
    s = requests.Session()
    passkey = get_parameters()
    s.auth = ('infoblox-backup', passkey)
    path = '/tmp/infoblox_backup-%s.bak' % (datetime.now().strftime('%Y%m%d'))
    backup_name = 'infoblox_backup-%s.bak' % (datetime.now().strftime('%Y%m%d_%s'))
    url = 'https://' + INFOBLOX_GRIDMASTER + '/wapi/v2.5/fileop?_function=getgriddata'
    headers = {'Content-type': 'application/json'}
    payload = {"type": "BACKUP"}

    """
    Create backup on Infoblox gridmaster
    """
    r = s.post(url, json=payload, headers=headers, verify=False)
    d = json.loads(r.text)
    token = d['token']
    backup_url = d['url']

    """
    Download backup from Infoblox gridmaster
    """
    s3_path = 'twitch-infoblox-backups/'
    r = s.get(backup_url, headers=headers, stream=True, verify=False)
    if r.status_code == 200:
        with open(path, 'wb') as f:
            for chunk in r.iter_content(1024):
                f.write(chunk)

    """
    Upload backup to s3 bucket
    """
    with open(path, 'rb') as data:
        bucket.upload_fileobj(data, s3_path + backup_name)

    """
    Delete backup from Infoblox gridmaster
    """
    url = 'https://' + INFOBLOX_GRIDMASTER + '/wapi/v2.5/fileop?_function=downloadcomplete'
    payload = {'token': token}
    r = s.post(url, json=payload, headers=headers, verify=False)

    return {
        "statusCode": 200,
        "body": json.dumps("backup success")
    }


if __name__ == "__main__":
    event = {'mode': 'cli'}
    lambda_handler(event, {})
