#!/usr/bin/env bash

set -eo pipefail

export AWS_STS_REGIONAL_ENDPOINTS=regional
export AWS_SDK_LOAD_CONFIG=true

usage() {
    mapfile -t valid_domains < <(find conf -type f| sed -r 's/.*\/(.*)\..*/\1/')

    cat <<EOF
Usage: $0 -s <service> -d <domain> <command> [args ...]

  <service>
    The name of the service subdomain. E.g. puppet


  <domain>
    The domain that this service subdomain should be delegated to.
    Valid domains are:
      $(printf '%s\n      ' "${valid_domains[@]}")

  <command> [args ...]
    In general, these are command and args that are passed to terraform binary as
    arguments

    E.g.
      $0 -s <service> -d <domain> plan

      gets passed to terraform as

      terraform plan

    Exception
    ---------

    'cleanup' is a special command which deletes existing delegation for
    <service> in <domain>.

    E.g.
      $0 -s <service> -d <domain> cleanup
EOF
    exit 1
}

requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && usage )
    command -v jq &> /dev/null || ( echo "jq required but missing" && usage )
}

tf_init() {
    shift
    terraform init "$@" -reconfigure -backend-config=<(cat <<EOF
key    = "services/${service}.tfstate"
bucket = "${TF_S3_STATE_BUCKET}"
EOF
)
}

validate_account() {
    if ! [[ "$AWS_ACCOUNT_ID" == "$(aws sts get-caller-identity | jq -r .Account)" ]]; then
        echo
        echo "Your current credentials do not match account '${AWS_ACCOUNT_ID}' that manages '${domain}' domain."
        echo "Please double check that you are using the correct credentials for the correct account."
        echo
        usage
    fi
}

check_service_exist() {
    [[ -f "services/${domain}/${service}.tfvars" ]] || (echo "The file 'services/${domain}/${service}.tfvars' has not been setup" && usage)
}

delete_records() {
    service_name=$1
    if [[ -n $service_name ]]; then
        nsrecordset="$(aws route53 list-resource-record-sets --hosted-zone-id "$ROUTE53_ZONE_ID" | jq '.ResourceRecordSets[] | select (.Name | contains("'"${service_name}.${domain}."'")) | select (.Type == "NS")')"
        txtrecordset="$(aws route53 list-resource-record-sets --hosted-zone-id "$ROUTE53_ZONE_ID" | jq '.ResourceRecordSets[] | select (.Name | contains("'"${service_name}.${domain}."'")) | select (.Type == "TXT")')"

        if [[ -z "$txtrecordset" ]] || [[ -z "$nsrecordset" ]]; then
            echo "No NS or TXT records for '$service_name' found"
            exit 1
        else
            aws route53 change-resource-record-sets --hosted-zone-id "$ROUTE53_ZONE_ID" --change-batch '{"Changes": [{"Action": "DELETE", "ResourceRecordSet": '"$nsrecordset"'}]}'
            aws route53 change-resource-record-sets --hosted-zone-id "$ROUTE53_ZONE_ID" --change-batch '{"Changes": [{"Action": "DELETE", "ResourceRecordSet": '"$txtrecordset"'}]}'
        fi

        terraform state rm data.aws_route53_zone.subdomain aws_route53_record.comment aws_route53_record.subdomain || true

        if git rm "services/${domain}/${service_name}.tfvars"; then
            echo "Please open a pull request to have this change merged"
        fi

        echo "Done"
    else
        echo "No service name found. Please try again with a legitimate service name"
        exit 1
    fi
}

confirmation_prompt() {
    read -rp "Are you sure you want to delete zone $1? [Yes|No]"$'\n' input
    case "$input" in
        [yY][eE][sS])
            echo "Starting record cleanup..."
            ;;
        [nN][oO])
            echo "Canceled record cleanup"
            exit 1
            ;;
        *)
            echo "Please type \"yes\" or \"no\""
            confirmation_prompt "$1"
            ;;
    esac
}

check_service_in_use() {
    service=$1
    set +e
    terraform plan -detailed-exitcode -var-file="services/${domain}/${service}.tfvars" &>/dev/null
    exit_code="$?"
    set -e
    case "$exit_code" in
    # Exit codes returned because of -detailed-exitcode:
    # 0 = Succeeded with empty diff (no changes)
    # 1 = Error
    # 2 = Succeeded with non-empty diff (changes present)
        0)
            printf "\nWARNING: This service's resources are still up and running - it may still be in use.\nIf you confirm you want to delete, the zone records will be destroyed.\n"
            confirmation_prompt "$service"
            delete_records "$service"
            ;;
        1|2)
            confirmation_prompt "$service"
            delete_records "$service"
            ;;
        *)
            echo "Encountered unknown Terraform exit code: $?"
            exit 1
            ;;
    esac
}

while getopts 'hs:d:' option; do
    case "$option" in
        s)
            service="$OPTARG"
            ;;
        d)
            domain="$OPTARG"
            ;;
        h)
            usage
            ;;
        *)
            ;;
    esac
done

shift $((OPTIND-1))

## Make sure everything is set
[[ -z "$service" || -z "$domain" ]] && usage

# Source zone configs
if [[ -f "conf/${domain}.conf" ]]; then
    # shellcheck source=/dev/null
    . "conf/${domain}.conf"
else
    echo "ERROR: Domain ${domain} is invalid"
    usage
fi

check_service_exist
requirements
validate_account

case "$1" in
    init)
        tf_init "$@"
        ;;
    plan|apply)
        tf_init "$@"
        exec terraform "$@" \
            -var-file="services/${domain}/${service}.tfvars" \
            -var="domain=${domain}" \
            -var="domain_route53_id=${ROUTE53_ZONE_ID}"
        ;;
    cleanup)
        tf_init "$@"
        check_service_in_use "$service"
        ;;
    *)
        exec terraform "$@"
        ;;
esac
